<?php
// app/Http/Controllers/Auth/RegisteredUserController.php
namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        $warehouses = Warehouse::where('is_active', true)->orderBy('name')->get();
        return view('auth.register', compact('warehouses'));
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'warehouse_code' => ['required', 'string', 'exists:warehouses,code'],
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'role' => ['required', 'in:2wheel,4wheel'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'terms' => ['required', 'accepted'],
        ], [
            'warehouse_code.required' => 'กรุณาเลือกคลัง',
            'warehouse_code.exists' => 'คลังที่เลือกไม่ถูกต้อง',
            'first_name.required' => 'กรุณากรอกชื่อ',
            'last_name.required' => 'กรุณากรอกนามสกุล',
            'role.required' => 'กรุณาเลือกประเภทรถ',
            'role.in' => 'ประเภทรถไม่ถูกต้อง',
            'email.required' => 'กรุณากรอกอีเมล',
            'email.email' => 'รูปแบบอีเมลไม่ถูกต้อง',
            'email.unique' => 'อีเมลนี้ถูกใช้งานแล้ว',
            'password.required' => 'กรุณากรอกรหัสผ่าน',
            'password.confirmed' => 'รหัสผ่านไม่ตรงกัน',
            'terms.required' => 'กรุณายอมรับข้อกำหนดและเงื่อนไข',
            'terms.accepted' => 'กรุณายอมรับข้อกำหนดและเงื่อนไข',
        ]);

        // ตรวจสอบว่าคลังยังใช้งานอยู่
        $warehouse = Warehouse::where('code', $request->warehouse_code)
                             ->where('is_active', true)
                             ->first();
        
        if (!$warehouse) {
            return back()->withErrors(['warehouse_code' => 'คลังที่เลือกไม่ได้ใช้งานในขณะนี้'])->withInput();
        }

        try {
            $user = User::create([
                'warehouse_code' => $request->warehouse_code,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'role' => $request->role,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'is_active' => true,
            ]);

            event(new Registered($user));
            Auth::login($user);

            return redirect(route('dashboard', absolute: false));
        } catch (\Exception $e) {
            \Log::error('Registration error: ' . $e->getMessage());
            return back()->withErrors(['email' => 'เกิดข้อผิดพลาดในการสร้างบัญชี กรุณาลองใหม่อีกครั้ง'])->withInput();
        }
    }
}