<?php

namespace App\Http\Controllers;

use App\Models\DeliveryRecord;
use App\Models\Warehouse;
use App\Models\CommissionRate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class DeliveryRecordController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        
        // ตรวจสอบสิทธิ์
        if (!in_array($user->role, ['2wheel', '4wheel'])) {
            abort(403, 'คุณไม่มีสิทธิ์เข้าถึงหน้านี้');
        }
        
        $selectedMonth = $request->input('month', now()->format('Y-m'));

        $query = DeliveryRecord::where('user_id', $user->id)
            ->whereRaw("DATE_FORMAT(delivery_date, '%Y-%m') = ?", [$selectedMonth])
            ->orderByDesc('delivery_date')
            ->orderByDesc('created_at');

        // กรอง keyword ถ้ามี
        if ($request->filled('search')) {
            $q = $request->input('search');
            $query->where(function ($sub) use ($q) {
                $sub->where('parcel_count', 'like', "%$q%")
                    ->orWhere('commission_amount', 'like', "%$q%");
            });
        }

        $records = $query->paginate(30)->withQueryString();

        // Summary สำหรับเดือนนั้น (ยอดรวม)
        $summary = [
            'total_days' => $query->count(),
            'total_parcels' => $query->sum('parcel_count'),
            'total_commission' => $query->sum('commission_amount'),
        ];

        // Weekly summary
        $weeklyData = [];
        foreach ($query->get() as $rec) {
            $week = date('W', strtotime($rec->delivery_date));
            if (!isset($weeklyData[$week])) {
                $weeklyData[$week] = ['parcels' => 0, 'commission' => 0, 'days' => 0];
            }
            $weeklyData[$week]['parcels'] += $rec->parcel_count;
            $weeklyData[$week]['commission'] += $rec->commission_amount;
            $weeklyData[$week]['days']++;
        }

        return view('vehicle.deliveries.index', [
            'records' => $records,
            'selectedMonth' => $selectedMonth,
            'summary' => $summary,
            'weeklyData' => $weeklyData,
        ]);
    }

    public function create()
    {
        $user = Auth::user();
        
        // ตรวจสอบสิทธิ์
        if (!in_array($user->role, ['2wheel', '4wheel'])) {
            abort(403, 'คุณไม่มีสิทธิ์เข้าถึงหน้านี้');
        }
        
        // ดึง commission rates สำหรับ user ปัจจุบัน (ใช้ warehouse_code แทน warehouse_id)
        $commissionRates = CommissionRate::where('warehouse_code', $user->warehouse_code ?? '')
            ->where('vehicle_type', $user->role)
            ->where('is_active', true)
            ->orderBy('tier')
            ->get();

        $user = $user->load('warehouse');
        
        return view('vehicle.deliveries.create', [
            'user' => $user,
            'commissionRates' => $commissionRates,
            'warehouse_code' => $user->warehouse_code ?? '',
        ]);
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        // ตรวจสอบสิทธิ์
        if (!in_array($user->role, ['2wheel', '4wheel'])) {
            abort(403, 'คุณไม่มีสิทธิ์เข้าถึงหน้านี้');
        }
        
        $request->validate([
            'delivery_date' => 'required|date',
            'parcel_count' => 'required|integer|min:1',
        ], [
            'delivery_date.required' => 'กรุณาระบุวันที่จัดส่ง',
            'parcel_count.required' => 'กรุณาระบุจำนวนพัสดุ',
        ]);

        // **ตรวจสอบซ้ำ**
        $exists = DeliveryRecord::where('user_id', $user->id)
            ->where('delivery_date', $request->delivery_date)
            ->exists();

        if ($exists) {
            return back()
                ->withErrors(['delivery_date' => 'คุณบันทึกข้อมูลสำหรับวันที่นี้แล้ว'])
                ->withInput();
        }

        // ดึง commission rates สำหรับ user
        $rates = CommissionRate::where('warehouse_code', $user->warehouse_code ?? '')
            ->where('vehicle_type', $user->role)
            ->where('is_active', true)
            ->orderBy('tier')
            ->get();

        // คำนวณค่าคอมมิชชั่น
        $commission = $this->calculateCommission($request->parcel_count, $rates);

        // บันทึก
        DeliveryRecord::create([
            'user_id' => $user->id,
            'warehouse_id' => $user->warehouse_id ?? null, // ถ้ามี warehouse_id
            'delivery_date' => $request->delivery_date,
            'parcel_count' => $request->parcel_count,
            'commission_amount' => $commission,
        ]);

        return redirect()->route('vehicle.dashboard')->with('success', 'บันทึกการจัดส่งสำเร็จ!');
    }

    public function edit(DeliveryRecord $delivery) 
    {
        $user = Auth::user();
        
        // ตรวจสอบสิทธิ์แก้ไข: เจ้าของ record เท่านั้น
        if ($user->id !== $delivery->user_id) {
            abort(403, 'คุณไม่มีสิทธิ์แก้ไขรายการนี้');
        }
        
        $warehouses = Warehouse::all();
        return view('vehicle.deliveries.edit', compact('delivery', 'warehouses'));
    }

    public function update(Request $request, DeliveryRecord $delivery) 
    {
        $user = Auth::user();
        
        // ตรวจสอบสิทธิ์
        if ($user->id !== $delivery->user_id) {
            abort(403, 'คุณไม่มีสิทธิ์แก้ไขรายการนี้');
        }
        
        $request->validate([
            'delivery_date' => 'required|date',
            'parcel_count' => 'required|integer|min:1',
        ]);

        // ดึง commission rates สำหรับ user
        $rates = CommissionRate::where('warehouse_code', $user->warehouse_code ?? '')
            ->where('vehicle_type', $user->role)
            ->where('is_active', true)
            ->orderBy('tier')
            ->get();

        $commission = $this->calculateCommission($request->parcel_count, $rates);

        $delivery->update([
            'delivery_date' => $request->delivery_date,
            'parcel_count' => $request->parcel_count,
            'commission_amount' => $commission,
        ]);
        
        return redirect()->route('vehicle.deliveries.index')->with('success', 'อัปเดตรายการสำเร็จ');
    }

    public function destroy(DeliveryRecord $delivery) 
    {
        $user = Auth::user();
        
        // ตรวจสอบสิทธิ์
        if ($user->id !== $delivery->user_id) {
            abort(403, 'คุณไม่มีสิทธิ์ลบรายการนี้');
        }
        
        $delivery->delete();
        return back()->with('success', 'ลบรายการสำเร็จ');
    }

    public function show(DeliveryRecord $delivery) 
    {
        $user = Auth::user();
        
        // ตรวจสอบสิทธิ์ดูข้อมูล
        if ($user->id !== $delivery->user_id) {
            abort(403, 'คุณไม่มีสิทธิ์เข้าดูรายการนี้');
        }
        
        return view('vehicle.deliveries.show', compact('delivery'));
    }

    public function previewCommission(Request $request)
    {
        $user = Auth::user();
        $parcelCount = intval($request->input('parcel_count', 0));

        // Debug: Log user info
        \Log::info('Commission Preview Debug', [
            'user_id' => $user->id,
            'user_role' => $user->role,
            'warehouse_code' => $user->warehouse_code ?? 'NULL',
            'parcel_count' => $parcelCount
        ]);

        $rates = CommissionRate::where('warehouse_code', $user->warehouse_code ?? '')
            ->where('vehicle_type', $user->role)
            ->where('is_active', true)
            ->orderBy('tier')
            ->get();

        // Debug: Log rates found
        \Log::info('Commission Rates Found', [
            'rates_count' => $rates->count(),
            'rates' => $rates->toArray()
        ]);

        $commission = 0;
        $tiers = [];
        
        if ($rates->isNotEmpty()) {
            $breakdown = $this->calculateCommissionBreakdown($parcelCount, $rates);
            $commission = $breakdown['total_commission'];
            $tiers = $breakdown['tiers'];
        }

        // Debug: Log final result
        \Log::info('Commission Calculation Result', [
            'commission' => $commission,
            'tiers' => $tiers
        ]);

        return response()->json([
            'commission' => $commission,
            'tiers' => $tiers,
            'debug' => [
                'user_warehouse_code' => $user->warehouse_code ?? 'NULL',
                'user_role' => $user->role,
                'rates_found' => $rates->count()
            ],
            'preview_html' => view('components.commission_preview', [
                'tiers' => $tiers,
                'commission' => $commission
            ])->render()
        ]);
    }

    // Private method to calculate commission based on your existing rates structure
    private function calculateCommission($parcelCount, $rates)
    {
        if ($parcelCount <= 0 || $rates->isEmpty()) {
            return 0;
        }

        $totalCommission = 0;
        $remainingParcels = $parcelCount;

        foreach ($rates as $rate) {
            if ($remainingParcels <= 0) {
                break;
            }

            // ใช้ชื่อฟิลด์ที่ถูกต้องตาม Seeder
            $minParcels = $rate->min_packages ?? 1;
            $maxParcels = $rate->max_packages ?? PHP_INT_MAX;
            $ratePerParcel = $rate->rate_per_package ?? 0;

            if ($parcelCount < $minParcels) {
                continue;
            }

            $parcelsInTier = min(
                $remainingParcels,
                $maxParcels - $minParcels + 1,
                $parcelCount - $minParcels + 1
            );

            if ($parcelsInTier > 0) {
                $totalCommission += $parcelsInTier * $ratePerParcel;
                $remainingParcels -= $parcelsInTier;
            }
        }

        return round($totalCommission, 2);
    }

    // Private method for detailed breakdown
    private function calculateCommissionBreakdown($parcelCount, $rates)
    {
        if ($parcelCount <= 0 || $rates->isEmpty()) {
            return [
                'total_commission' => 0,
                'tiers' => [],
                'remaining_parcels' => $parcelCount
            ];
        }

        $totalCommission = 0;
        $remainingParcels = $parcelCount;
        $tiers = [];

        foreach ($rates as $rate) {
            if ($remainingParcels <= 0) {
                break;
            }

            // ใช้ชื่อฟิลด์ที่ถูกต้องตาม Seeder
            $minParcels = $rate->min_packages ?? 1;
            $maxParcels = $rate->max_packages ?? PHP_INT_MAX;
            $ratePerParcel = $rate->rate_per_package ?? 0;

            if ($parcelCount < $minParcels) {
                continue;
            }

            $parcelsInTier = min(
                $remainingParcels,
                $maxParcels - $minParcels + 1,
                $parcelCount - $minParcels + 1
            );

            if ($parcelsInTier > 0) {
                $tierCommission = $parcelsInTier * $ratePerParcel;
                $totalCommission += $tierCommission;
                
                $tiers[] = [
                    'range' => $maxParcels === PHP_INT_MAX ? "{$minParcels}+" : "{$minParcels}-{$maxParcels}",
                    'min_parcels' => $minParcels,
                    'max_parcels' => $maxParcels === PHP_INT_MAX ? null : $maxParcels,
                    'count' => $parcelsInTier,
                    'rate' => $ratePerParcel,
                    'sum' => $tierCommission
                ];

                $remainingParcels -= $parcelsInTier;
            }
        }

        return [
            'total_commission' => round($totalCommission, 2),
            'tiers' => $tiers,
            'remaining_parcels' => $remainingParcels
        ];
    }

    public function validateDeliveryDate(Request $request)
    {
        $user = Auth::user();
        $date = $request->delivery_date;
        $deliveryId = $request->delivery_id; // สำหรับการแก้ไข

        $query = DeliveryRecord::where('user_id', $user->id)
            ->where('delivery_date', $date);

        if ($deliveryId) {
            $query->where('id', '!=', $deliveryId);
        }

        $exists = $query->exists();

        return response()->json([
            'valid' => !$exists,
            'message' => $exists ? 'คุณบันทึกข้อมูลสำหรับวันที่นี้แล้ว' : 'วันที่ใช้งานได้'
        ]);
    }
}