<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class SavingGoal extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'goal_name',
        'target_amount',
        'current_amount',
        'target_date',
        'status',
    ];

    protected $casts = [
        'target_amount' => 'decimal:2',
        'current_amount' => 'decimal:2',
        'target_date' => 'date',
    ];

    /**
     * Relationships
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Attributes
     */
    public function getRemainingAmountAttribute()
    {
        return max(0, $this->target_amount - $this->current_amount);
    }

    public function getProgressPercentageAttribute()
    {
        if ($this->target_amount <= 0) return 0;
        return min(100, round(($this->current_amount / $this->target_amount) * 100, 1));
    }

    public function getDaysRemainingAttribute()
    {
        if (!$this->target_date) return null;
        
        $days = Carbon::now()->diffInDays($this->target_date, false);
        return max(0, $days);
    }

    public function getMonthsRemainingAttribute()
    {
        if (!$this->target_date) return null;
        
        $months = Carbon::now()->diffInMonths($this->target_date, false);
        return max(0, $months);
    }

    public function getSuggestedMonthlySavingAttribute()
    {
        $months = $this->months_remaining;
        if (!$months || $months <= 0) return 0;
        
        return round($this->remaining_amount / $months, 2);
    }

    public function getIsCompletedAttribute()
    {
        return $this->status === 'completed' || $this->current_amount >= $this->target_amount;
    }

    public function getIsOverdueAttribute()
    {
        return $this->target_date && Carbon::now()->isAfter($this->target_date) && !$this->is_completed;
    }

    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'active' => $this->is_overdue ? '#ef4444' : '#3b82f6',
            'completed' => '#10b981',
            'cancelled' => '#6b7280',
            default => '#6b7280'
        };
    }

    public function getStatusTextAttribute()
    {
        if ($this->is_completed) return 'เป้าหมายสำเร็จ';
        if ($this->is_overdue) return 'เกินกำหนด';
        
        return match($this->status) {
            'active' => 'กำลังออม',
            'completed' => 'สำเร็จ',
            'cancelled' => 'ยกเลิก',
            default => 'ไม่ทราบ'
        };
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Methods
     */
    public function deposit($amount)
    {
        $this->increment('current_amount', $amount);
        
        // Auto complete if reached target
        if ($this->current_amount >= $this->target_amount) {
            $this->update(['status' => 'completed']);
        }
        
        return $this;
    }

    public function withdraw($amount)
    {
        $this->decrement('current_amount', min($amount, $this->current_amount));
        
        // Reactivate if was completed
        if ($this->status === 'completed' && $this->current_amount < $this->target_amount) {
            $this->update(['status' => 'active']);
        }
        
        return $this;
    }

    public function complete()
    {
        $this->update([
            'status' => 'completed',
            'current_amount' => $this->target_amount
        ]);
        
        return $this;
    }

    public function cancel()
    {
        $this->update(['status' => 'cancelled']);
        
        return $this;
    }

    public function reactivate()
    {
        $this->update(['status' => 'active']);
        
        return $this;
    }

    public static function getSummaryStats($userId)
    {
        $goals = self::forUser($userId)->get();
        
        return [
            'total_goals' => $goals->count(),
            'active_goals' => $goals->where('status', 'active')->count(),
            'completed_goals' => $goals->where('status', 'completed')->count(),
            'total_target' => $goals->where('status', 'active')->sum('target_amount'),
            'total_saved' => $goals->where('status', 'active')->sum('current_amount'),
            'overall_progress' => $goals->where('status', 'active')->count() > 0 
                ? round($goals->where('status', 'active')->avg('progress_percentage'), 1) 
                : 0,
        ];
    }
}