<?php
// database/migrations/2025_07_27_142225_fix_commission_rates_table.php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Check if table exists, if not create it
        if (!Schema::hasTable('commission_rates')) {
            Schema::create('commission_rates', function (Blueprint $table) {
                $table->id();
                $table->string('warehouse_code', 10)->index();
                $table->enum('vehicle_type', ['2wheel', '4wheel'])->index();
                $table->integer('tier')->default(1);
                $table->integer('min_packages')->default(1);
                $table->integer('max_packages')->nullable();
                $table->decimal('rate_per_package', 8, 2)->default(0);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                
                // Foreign key constraint (only if warehouses table exists)
                if (Schema::hasTable('warehouses')) {
                    $table->foreign('warehouse_code')->references('code')->on('warehouses')->onDelete('cascade');
                }
                
                // Unique constraint to prevent duplicate tiers
                $table->unique(['warehouse_code', 'vehicle_type', 'tier'], 'unique_warehouse_vehicle_tier');
                
                // Index for faster queries
                $table->index(['warehouse_code', 'vehicle_type', 'is_active'], 'idx_warehouse_vehicle_active');
            });
        } else {
            // If table exists, check and update column names if needed
            $this->updateExistingTable();
        }
    }

    /**
     * Update existing table structure
     */
    private function updateExistingTable(): void
    {
        // Get existing columns
        $columns = $this->getTableColumns('commission_rates');
        
        Schema::table('commission_rates', function (Blueprint $table) use ($columns) {
            // Check if old column names exist and rename them
            if (in_array('min_amount', $columns) && !in_array('min_packages', $columns)) {
                $table->renameColumn('min_amount', 'min_packages');
            }
            
            if (in_array('max_amount', $columns) && !in_array('max_packages', $columns)) {
                $table->renameColumn('max_amount', 'max_packages');
            }
            
            if (in_array('rate', $columns) && !in_array('rate_per_package', $columns)) {
                $table->renameColumn('rate', 'rate_per_package');
            }
        });

        // Add missing columns
        Schema::table('commission_rates', function (Blueprint $table) use ($columns) {
            if (!in_array('tier', $columns)) {
                $table->integer('tier')->default(1)->after('vehicle_type');
            }
            
            if (!in_array('is_active', $columns)) {
                $table->boolean('is_active')->default(true);
            }
        });

        // Add indexes if they don't exist
        $this->addMissingIndexes();
    }

    /**
     * Get table columns
     */
    private function getTableColumns(string $tableName): array
    {
        try {
            return Schema::getColumnListing($tableName);
        } catch (\Exception $e) {
            return [];
        }
    }

    /**
     * Add missing indexes
     */
    private function addMissingIndexes(): void
    {
        $existingIndexes = $this->getExistingIndexes('commission_rates');
        
        // Add warehouse_code index if not exists
        if (!$this->indexExists($existingIndexes, 'warehouse_code')) {
            try {
                Schema::table('commission_rates', function (Blueprint $table) {
                    $table->index('warehouse_code');
                });
            } catch (\Exception $e) {
                // Index might already exist, skip
            }
        }
        
        // Add vehicle_type index if not exists
        if (!$this->indexExists($existingIndexes, 'vehicle_type')) {
            try {
                Schema::table('commission_rates', function (Blueprint $table) {
                    $table->index('vehicle_type');
                });
            } catch (\Exception $e) {
                // Index might already exist, skip
            }
        }
        
        // Add unique constraint if not exists
        if (!$this->indexExists($existingIndexes, 'unique_warehouse_vehicle_tier')) {
            try {
                Schema::table('commission_rates', function (Blueprint $table) {
                    $table->unique(['warehouse_code', 'vehicle_type', 'tier'], 'unique_warehouse_vehicle_tier');
                });
            } catch (\Exception $e) {
                // Constraint might already exist or data conflicts, skip
            }
        }
        
        // Add composite index if not exists
        if (!$this->indexExists($existingIndexes, 'idx_warehouse_vehicle_active')) {
            try {
                Schema::table('commission_rates', function (Blueprint $table) {
                    $table->index(['warehouse_code', 'vehicle_type', 'is_active'], 'idx_warehouse_vehicle_active');
                });
            } catch (\Exception $e) {
                // Index might already exist, skip
            }
        }
    }

    /**
     * Get existing indexes for a table
     */
    private function getExistingIndexes(string $tableName): array
    {
        try {
            $connection = Schema::getConnection();
            $databaseName = $connection->getDatabaseName();
            
            $indexes = DB::select("
                SELECT DISTINCT INDEX_NAME as name 
                FROM INFORMATION_SCHEMA.STATISTICS 
                WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ?
            ", [$databaseName, $tableName]);
            
            return array_column($indexes, 'name');
        } catch (\Exception $e) {
            return [];
        }
    }

    /**
     * Check if index exists
     */
    private function indexExists(array $existingIndexes, string $indexName): bool
    {
        return in_array($indexName, $existingIndexes);
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // For safety, we won't drop existing tables in rollback
        // Only remove indexes we added if they exist
        
        if (Schema::hasTable('commission_rates')) {
            $existingIndexes = $this->getExistingIndexes('commission_rates');
            
            Schema::table('commission_rates', function (Blueprint $table) use ($existingIndexes) {
                // Remove custom indexes we added
                if ($this->indexExists($existingIndexes, 'unique_warehouse_vehicle_tier')) {
                    try {
                        $table->dropUnique('unique_warehouse_vehicle_tier');
                    } catch (\Exception $e) {
                        // Index might not exist or be named differently
                    }
                }
                
                if ($this->indexExists($existingIndexes, 'idx_warehouse_vehicle_active')) {
                    try {
                        $table->dropIndex('idx_warehouse_vehicle_active');
                    } catch (\Exception $e) {
                        // Index might not exist or be named differently
                    }
                }
            });
        }
    }
};