@extends('layouts.app')

@section('title', 'จัดการผู้ใช้งาน - ระบบจัดการค่าคอมมิชชั่น')

@section('content')
<div class="content-container">
    <!-- Enhanced Page Header -->
    <div class="page-header">
        <div class="page-header-content">
            <div class="page-header-text">
                <h1 class="page-title">
                    <i class="fas fa-users text-orange-500"></i>
                    <span>จัดการผู้ใช้งาน</span>
                </h1>
                <p class="page-subtitle">จัดการข้อมูลผู้ใช้งานทั้งหมดในระบบ</p>
            </div>
            <div class="page-header-actions">
                <button onclick="exportUsers()" class="btn btn-info btn-sm-full">
                    <i class="fas fa-download"></i>
                    <span class="btn-text">Export CSV</span>
                </button>
                <a href="{{ route('admin.users.create') }}" class="btn btn-primary btn-sm-full">
                    <i class="fas fa-user-plus"></i>
                    <span class="btn-text">เพิ่มผู้ใช้ใหม่</span>
                </a>
            </div>
        </div>
    </div>

    <!-- Enhanced Stats Cards -->
    <div class="dashboard-grid">
        <div class="dashboard-card dashboard-card-orange">
            <div class="dashboard-card-hover"></div>
            <div class="dashboard-card-main">
                <div class="dashboard-card-row">
                    <div class="dashboard-card-icon dashboard-card-icon-orange">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="dashboard-card-info">
                        <div class="dashboard-card-num">{{ number_format($stats['total'] ?? 0) }}</div>
                        <div class="dashboard-card-label">ผู้ใช้ทั้งหมด</div>
                    </div>
                </div>
                <div class="dashboard-card-row2">
                    <div class="dashboard-card-status">
                        <i class="fas fa-chart-line"></i>
                        <span>ทั้งระบบ</span>
                    </div>
                    <a href="#" class="dashboard-card-link">ดูรายละเอียด</a>
                </div>
            </div>
        </div>

        <div class="dashboard-card dashboard-card-green">
            <div class="dashboard-card-hover"></div>
            <div class="dashboard-card-main">
                <div class="dashboard-card-row">
                    <div class="dashboard-card-icon dashboard-card-icon-green">
                        <i class="fas fa-user-check"></i>
                    </div>
                    <div class="dashboard-card-info">
                        <div class="dashboard-card-num">{{ number_format($stats['active'] ?? 0) }}</div>
                        <div class="dashboard-card-label">ใช้งานอยู่</div>
                    </div>
                </div>
                <div class="dashboard-card-row2">
                    <div class="dashboard-card-status dashboard-card-status-green">
                        <i class="fas fa-check"></i>
                        <span>Active</span>
                    </div>
                    <a href="#" onclick="filterByStatus('active')" class="dashboard-card-link">กรองดู</a>
                </div>
            </div>
        </div>

        <div class="dashboard-card dashboard-card-blue">
            <div class="dashboard-card-hover"></div>
            <div class="dashboard-card-main">
                <div class="dashboard-card-row">
                    <div class="dashboard-card-icon dashboard-card-icon-blue">
                        <i class="fas fa-motorcycle"></i>
                    </div>
                    <div class="dashboard-card-info">
                        <div class="dashboard-card-num">{{ number_format($stats['2wheel'] ?? 0) }}</div>
                        <div class="dashboard-card-label">รถ 2 ล้อ</div>
                    </div>
                </div>
                <div class="dashboard-card-row2">
                    <div class="dashboard-card-status dashboard-card-status-blue">
                        <i class="fas fa-motorcycle"></i>
                        <span>2-Wheeler</span>
                    </div>
                    <a href="#" onclick="filterByRole('2wheel')" class="dashboard-card-link">กรองดู</a>
                </div>
            </div>
        </div>

        <div class="dashboard-card dashboard-card-purple">
            <div class="dashboard-card-hover"></div>
            <div class="dashboard-card-main">
                <div class="dashboard-card-row">
                    <div class="dashboard-card-icon dashboard-card-icon-purple">
                        <i class="fas fa-truck"></i>
                    </div>
                    <div class="dashboard-card-info">
                        <div class="dashboard-card-num">{{ number_format($stats['4wheel'] ?? 0) }}</div>
                        <div class="dashboard-card-label">รถ 4 ล้อ</div>
                    </div>
                </div>
                <div class="dashboard-card-row2">
                    <div class="dashboard-card-status dashboard-card-status-purple">
                        <i class="fas fa-truck"></i>
                        <span>4-Wheeler</span>
                    </div>
                    <a href="#" onclick="filterByRole('4wheel')" class="dashboard-card-link">กรองดู</a>
                </div>
            </div>
        </div>
    </div>

    <!-- Enhanced Filters -->
    <div class="modern-card">
        <div class="modern-card-header">
            <h3 class="modern-card-title">
                <i class="fas fa-filter text-blue-500"></i>
                ตัวกรองและค้นหา
            </h3>
            <!-- Mobile Filter Toggle -->
            <button class="mobile-filter-toggle" onclick="toggleFilters()">
                <i class="fas fa-sliders-h"></i>
            </button>
        </div>
        <div class="modern-card-body" id="filterBody">
            <form method="GET" action="{{ route('admin.users.index') }}" class="filter-form">
                <div class="filter-grid">
                    <div class="form-group">
                        <label class="form-label">ค้นหา</label>
                        <div class="input-with-icon">
                            <i class="fas fa-search input-icon"></i>
                            <input type="text" name="search" value="{{ request('search') }}" 
                                   class="form-control" placeholder="ชื่อ, อีเมล หรือรหัสคลัง">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">บทบาท</label>
                        <div class="input-with-icon">
                            <i class="fas fa-user-tag input-icon"></i>
                            <select name="role" class="form-control">
                                <option value="">ทั้งหมด</option>
                                <option value="admin" {{ request('role') === 'admin' ? 'selected' : '' }}>
                                    👑 ผู้ดูแลระบบ
                                </option>
                                <option value="2wheel" {{ request('role') === '2wheel' ? 'selected' : '' }}>
                                    🏍️ รถ 2 ล้อ
                                </option>
                                <option value="4wheel" {{ request('role') === '4wheel' ? 'selected' : '' }}>
                                    🚚 รถ 4 ล้อ
                                </option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">สถานะ</label>
                        <div class="input-with-icon">
                            <i class="fas fa-toggle-on input-icon"></i>
                            <select name="status" class="form-control">
                                <option value="">ทั้งหมด</option>
                                <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>
                                    ✅ ใช้งานอยู่
                                </option>
                                <option value="inactive" {{ request('status') === 'inactive' ? 'selected' : '' }}>
                                    ❌ ปิดใช้งาน
                                </option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label d-mobile-none">&nbsp;</label>
                        <div class="filter-actions">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-search"></i>ค้นหา
                            </button>
                            <a href="{{ route('admin.users.index') }}" class="btn btn-secondary">
                                <i class="fas fa-times"></i>ล้าง
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Bulk Actions -->
    <div class="modern-card bulk-actions-card" id="bulkActions" style="display: none;">
        <div class="modern-card-body">
            <div class="bulk-actions-content">
                <div class="bulk-info">
                    <div class="bulk-count">
                        <i class="fas fa-check-circle"></i>
                        เลือกแล้ว <span id="selectedCount">0</span> รายการ
                    </div>
                </div>
                <div class="bulk-buttons">
                    <button onclick="bulkAction('activate')" class="btn btn-success btn-sm">
                        <i class="fas fa-user-check"></i>
                        <span class="d-mobile-none">เปิดใช้งาน</span>
                    </button>
                    <button onclick="bulkAction('deactivate')" class="btn btn-warning btn-sm">
                        <i class="fas fa-user-slash"></i>
                        <span class="d-mobile-none">ปิดใช้งาน</span>
                    </button>
                    <button onclick="clearSelection()" class="btn btn-secondary btn-sm">
                        <i class="fas fa-times"></i>
                        <span class="d-mobile-none">ยกเลิก</span>
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Enhanced Users Table -->
    <div class="modern-card">
        <div class="modern-card-header">
            <div class="card-header-content">
                <h3 class="modern-card-title">
                    <i class="fas fa-list text-green-500"></i>
                    รายชื่อผู้ใช้งาน
                </h3>
                <div class="card-count">
                    ทั้งหมด {{ ($users ?? collect())->total() ?? 0 }} รายการ
                </div>
            </div>
        </div>
        
        <div class="modern-card-body p-0">
            <!-- Desktop Table View -->
            <div class="table-responsive desktop-view">
                <table class="modern-table">
                    <thead>
                        <tr>
                            <th class="checkbox-col">
                                <input type="checkbox" id="selectAll" 
                                       class="modern-checkbox">
                            </th>
                            <th>ผู้ใช้</th>
                            <th>บทบาท</th>
                            <th>คลังสินค้า</th>
                            <th>สถานะ</th>
                            <th>สร้างเมื่อ</th>
                            <th class="actions-col">การดำเนินการ</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($users ?? [] as $user)
                        <tr class="table-row">
                            <td>
                                <input type="checkbox" class="user-checkbox modern-checkbox" 
                                       value="{{ $user->id }}">
                            </td>
                            <td>
                                <div class="user-info">
                                    <div class="user-avatar">
                                        {{ strtoupper(substr($user->name ?? 'U', 0, 2)) }}
                                    </div>
                                    <div class="user-details">
                                        <div class="user-name">{{ $user->name ?? 'ไม่ระบุชื่อ' }}</div>
                                        <div class="user-email">{{ $user->email ?? 'ไม่ระบุอีเมล' }}</div>
                                    </div>
                                </div>
                            </td>
                            <td>
                                <span class="role-badge role-{{ $user->role ?? 'unknown' }}">
                                    @if(($user->role ?? '') === 'admin')
                                        <i class="fas fa-crown"></i>
                                        <span class="d-mobile-none">ผู้ดูแลระบบ</span>
                                        <span class="d-desktop-none">Admin</span>
                                    @elseif(($user->role ?? '') === '2wheel')
                                        <i class="fas fa-motorcycle"></i>
                                        <span class="d-mobile-none">รถ 2 ล้อ</span>
                                        <span class="d-desktop-none">2W</span>
                                    @elseif(($user->role ?? '') === '4wheel')
                                        <i class="fas fa-truck"></i>
                                        <span class="d-mobile-none">รถ 4 ล้อ</span>
                                        <span class="d-desktop-none">4W</span>
                                    @else
                                        <i class="fas fa-question"></i>
                                        {{ $user->role ?? 'ไม่ระบุ' }}
                                    @endif
                                </span>
                            </td>
                            <td>
                                <div class="warehouse-info">
                                    @if($user->warehouse_code)
                                        <div class="warehouse-code">{{ $user->warehouse_code }}</div>
                                        <div class="warehouse-name">{{ $user->warehouse_name ?? 'ไม่พบชื่อคลัง' }}</div>
                                    @else
                                        <span class="text-gray-400">-</span>
                                    @endif
                                </div>
                            </td>
                            <td>
                                <span class="status-badge status-{{ ($user->is_active ?? false) ? 'active' : 'inactive' }}">
                                    @if($user->is_active ?? false)
                                        <i class="fas fa-check-circle"></i>
                                        <span class="d-mobile-none">ใช้งานอยู่</span>
                                    @else
                                        <i class="fas fa-times-circle"></i>
                                        <span class="d-mobile-none">ปิดใช้งาน</span>
                                    @endif
                                </span>
                            </td>
                            <td>
                                <div class="date-info">
                                    {{ $user->created_at ? $user->created_at->format('d/m/Y') : '-' }}
                                    @if($user->created_at)
                                        <div class="time-info">{{ $user->created_at->format('H:i') }}</div>
                                    @endif
                                </div>
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <a href="{{ route('admin.users.edit', $user) }}" 
                                       class="action-btn action-edit" title="แก้ไข">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    @if($user->id !== auth()->id())
                                        <button onclick="toggleUserStatus({{ $user->id }})" 
                                                class="action-btn action-toggle" 
                                                title="{{ ($user->is_active ?? false) ? 'ปิดใช้งาน' : 'เปิดใช้งาน' }}">
                                            <i class="fas fa-{{ ($user->is_active ?? false) ? 'user-slash' : 'user-check' }}"></i>
                                        </button>
                                        <button onclick="resetPassword({{ $user->id }})" 
                                                class="action-btn action-reset" title="รีเซ็ตรหัสผ่าน">
                                            <i class="fas fa-key"></i>
                                        </button>
                                        <button onclick="deleteUser({{ $user->id }})" 
                                                class="action-btn action-delete" title="ลบผู้ใช้">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    @else
                                        <span class="text-xs text-gray-400 px-2">ตัวคุณเอง</span>
                                    @endif
                                </div>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="7" class="empty-state">
                                <div class="empty-content">
                                    <i class="fas fa-users text-6xl text-gray-300 mb-4"></i>
                                    <h3 class="text-xl font-semibold text-gray-500 mb-2">ไม่มีผู้ใช้งาน</h3>
                                    <p class="text-gray-400 mb-4">ยังไม่มีผู้ใช้งานในระบบ หรือไม่พบผลการค้นหา</p>
                                    <a href="{{ route('admin.users.create') }}" class="btn btn-primary">
                                        <i class="fas fa-plus"></i>เพิ่มผู้ใช้แรก
                                    </a>
                                </div>
                            </td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            <!-- Mobile Card View -->
            <div class="mobile-view">
                @forelse($users ?? [] as $user)
                <div class="mobile-card" data-user-id="{{ $user->id }}">
                    <div class="mobile-card-header">
                        <input type="checkbox" class="user-checkbox modern-checkbox" 
                               value="{{ $user->id }}">
                        <div class="user-info">
                            <div class="user-avatar">
                                {{ strtoupper(substr($user->name ?? 'U', 0, 2)) }}
                            </div>
                            <div class="user-details">
                                <div class="user-name">{{ $user->name ?? 'ไม่ระบุชื่อ' }}</div>
                                <div class="user-email">{{ $user->email ?? 'ไม่ระบุอีเมล' }}</div>
                            </div>
                        </div>
                        @if($user->id !== auth()->id())
                        <button class="mobile-menu-btn" onclick="toggleMobileMenu({{ $user->id }})">
                            <i class="fas fa-ellipsis-v"></i>
                        </button>
                        @endif
                    </div>
                    
                    <div class="mobile-card-body">
                        <div class="mobile-info-row">
                            <span class="info-label">บทบาท:</span>
                            <span class="role-badge role-{{ $user->role ?? 'unknown' }}">
                                @if(($user->role ?? '') === 'admin')
                                    <i class="fas fa-crown"></i> Admin
                                @elseif(($user->role ?? '') === '2wheel')
                                    <i class="fas fa-motorcycle"></i> 2 ล้อ
                                @elseif(($user->role ?? '') === '4wheel')
                                    <i class="fas fa-truck"></i> 4 ล้อ
                                @else
                                    {{ $user->role ?? 'ไม่ระบุ' }}
                                @endif
                            </span>
                        </div>
                        
                        <div class="mobile-info-row">
                            <span class="info-label">คลัง:</span>
                            <span class="info-value">
                                @if($user->warehouse_code)
                                    {{ $user->warehouse_code }}
                                @else
                                    -
                                @endif
                            </span>
                        </div>
                        
                        <div class="mobile-info-row">
                            <span class="info-label">สถานะ:</span>
                            <span class="status-badge status-{{ ($user->is_active ?? false) ? 'active' : 'inactive' }}">
                                @if($user->is_active ?? false)
                                    <i class="fas fa-check-circle"></i> ใช้งาน
                                @else
                                    <i class="fas fa-times-circle"></i> ปิดใช้งาน
                                @endif
                            </span>
                        </div>
                        
                        <div class="mobile-info-row">
                            <span class="info-label">สร้างเมื่อ:</span>
                            <span class="info-value">
                                {{ $user->created_at ? $user->created_at->format('d/m/Y H:i') : '-' }}
                            </span>
                        </div>
                    </div>
                    
                    @if($user->id !== auth()->id())
                    <div class="mobile-actions" id="mobile-actions-{{ $user->id }}" style="display: none;">
                        <a href="{{ route('admin.users.edit', $user) }}" class="mobile-action-btn action-edit">
                            <i class="fas fa-edit"></i> แก้ไข
                        </a>
                        <button onclick="toggleUserStatus({{ $user->id }})" class="mobile-action-btn action-toggle">
                            <i class="fas fa-{{ ($user->is_active ?? false) ? 'user-slash' : 'user-check' }}"></i>
                            {{ ($user->is_active ?? false) ? 'ปิดใช้งาน' : 'เปิดใช้งาน' }}
                        </button>
                        <button onclick="resetPassword({{ $user->id }})" class="mobile-action-btn action-reset">
                            <i class="fas fa-key"></i> รีเซ็ตรหัสผ่าน
                        </button>
                        <button onclick="deleteUser({{ $user->id }})" class="mobile-action-btn action-delete">
                            <i class="fas fa-trash"></i> ลบ
                        </button>
                    </div>
                    @endif
                </div>
                @empty
                <div class="empty-state-mobile">
                    <i class="fas fa-users"></i>
                    <h3>ไม่มีผู้ใช้งาน</h3>
                    <p>ยังไม่มีผู้ใช้งานในระบบ</p>
                    <a href="{{ route('admin.users.create') }}" class="btn btn-primary btn-sm">
                        <i class="fas fa-plus"></i>เพิ่มผู้ใช้แรก
                    </a>
                </div>
                @endforelse
            </div>
        </div>
        
        <!-- Enhanced Pagination -->
        @if(isset($users) && $users->hasPages())
        <div class="modern-card-footer">
            <div class="pagination-container">
                <div class="pagination-info">
                    แสดง {{ $users->firstItem() ?? 0 }} ถึง {{ $users->lastItem() ?? 0 }} 
                    จาก {{ $users->total() ?? 0 }} รายการ
                </div>
                <div class="pagination-wrapper">
                    {{ $users->links() }}
                </div>
            </div>
        </div>
        @endif
    </div>
</div>
@endsection

@push('styles')
<style>
/* Root Variables */
:root {
    --mobile-breakpoint: 768px;
    --tablet-breakpoint: 1024px;
    --desktop-breakpoint: 1280px;
}

/* Content Container */
.content-container {
    padding: 1.5rem;
    max-width: 100%;
    margin: 0 auto;
}

/* Page Header */
.page-header {
    background: var(--white);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    margin-bottom: 1.5rem;
    box-shadow: var(--shadow-sm);
    border: 1px solid var(--medium-gray);
}

.page-header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

.page-header-text {
    flex: 1;
    min-width: 200px;
}

.page-title {
    font-size: 1.5rem;
    font-weight: 600;
    color: var(--dark-gray);
    margin-bottom: 0.25rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.page-subtitle {
    color: var(--gray);
    font-size: 0.9rem;
    margin: 0;
}

.page-header-actions {
    display: flex;
    gap: 0.75rem;
    flex-wrap: wrap;
}

/* Modern Card */
.modern-card {
    background: var(--white);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-sm);
    border: 1px solid var(--medium-gray);
    margin-bottom: 1.5rem;
    overflow: hidden;
}

.modern-card-header {
    background: linear-gradient(135deg, rgba(255, 107, 53, 0.05) 0%, rgba(255, 140, 97, 0.05) 100%);
    padding: 1rem 1.5rem;
    border-bottom: 1px solid var(--medium-gray);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.card-header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    width: 100%;
    flex-wrap: wrap;
    gap: 0.5rem;
}

.modern-card-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--dark-gray);
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin: 0;
}

.card-count {
    color: var(--gray);
    font-size: 0.875rem;
}

.modern-card-body {
    padding: 1.5rem;
}

.modern-card-body.p-0 {
    padding: 0;
}

/* Dashboard Grid */
.dashboard-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
    margin-bottom: 1.5rem;
}

/* Filter Section */
.mobile-filter-toggle {
    display: none;
    background: none;
    border: none;
    color: var(--primary-orange);
    font-size: 1.25rem;
    cursor: pointer;
    padding: 0.5rem;
}

.filter-form {
    width: 100%;
}

.filter-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    align-items: end;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.form-label {
    font-weight: 500;
    color: var(--dark-gray);
    font-size: 0.875rem;
}

.input-with-icon {
    position: relative;
    display: flex;
    align-items: center;
}

.input-icon {
    position: absolute;
    left: 0.75rem;
    color: var(--gray);
    font-size: 0.875rem;
    z-index: 1;
}

.form-control {
    width: 100%;
    padding: 0.625rem 1rem 0.625rem 2.25rem;
    border: 2px solid var(--medium-gray);
    border-radius: var(--radius-md);
    font-family: inherit;
    font-size: 0.9rem;
    transition: var(--transition-fast);
    background: var(--white);
}

.filter-actions {
    display: flex;
    gap: 0.5rem;
}

/* Bulk Actions */
.bulk-actions-card {
    position: sticky;
    top: 1rem;
    z-index: 10;
}

.bulk-actions-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

.bulk-info {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.bulk-count {
    font-size: 1rem;
    font-weight: 600;
    color: var(--primary-orange);
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.bulk-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

/* Buttons */
.btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    padding: 0.625rem 1.25rem;
    border: none;
    border-radius: var(--radius-md);
    font-family: inherit;
    font-weight: 500;
    font-size: 0.9rem;
    cursor: pointer;
    transition: var(--transition-fast);
    text-decoration: none;
    white-space: nowrap;
}

.btn-sm {
    padding: 0.5rem 1rem;
    font-size: 0.875rem;
}

.btn-primary {
    background: var(--primary-gradient);
    color: var(--white);
    box-shadow: var(--shadow-sm);
}

.btn-primary:hover {
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

.btn-secondary {
    background: #6B7280;
    color: white;
}

.btn-secondary:hover {
    background: #4B5563;
}

.btn-info {
    background: linear-gradient(135deg, #0ea5e9 0%, #0284c7 100%);
    color: white;
}

.btn-success {
    background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
    color: white;
}

.btn-warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    color: white;
}

/* Table Responsive */
.table-responsive {
    overflow-x: auto;
    -webkit-overflow-scrolling: touch;
}

.modern-table {
    width: 100%;
    min-width: 800px;
    border-collapse: collapse;
    background: white;
}

.modern-table th {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    padding: 0.875rem 1rem;
    text-align: left;
    font-weight: 600;
    font-size: 0.8rem;
    color: var(--dark-gray);
    text-transform: uppercase;
    letter-spacing: 0.05em;
    border-bottom: 2px solid var(--medium-gray);
}

.modern-table td {
    padding: 1rem;
    border-bottom: 1px solid #f1f5f9;
    vertical-align: middle;
}

.checkbox-col {
    width: 50px;
}

.actions-col {
    width: 150px;
}

/* User Info */
.user-info {
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.user-avatar {
    width: 40px;
    height: 40px;
    border-radius: 10px;
    background: var(--primary-gradient);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
    font-size: 0.9rem;
    flex-shrink: 0;
}

.user-details {
    flex: 1;
    min-width: 0;
}

.user-name {
    font-weight: 600;
    color: var(--dark-gray);
    font-size: 0.9rem;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.user-email {
    color: var(--gray);
    font-size: 0.8rem;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

/* Badges */
.role-badge, .status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.375rem;
    padding: 0.375rem 0.75rem;
    border-radius: var(--radius-full);
    font-size: 0.8rem;
    font-weight: 500;
    white-space: nowrap;
}

.role-admin {
    background: linear-gradient(135deg, #ede9fe 0%, #ddd6fe 100%);
    color: #7c3aed;
    border: 1px solid #c4b5fd;
}

.role-2wheel {
    background: linear-gradient(135deg, #dbeafe 0%, #bfdbfe 100%);
    color: #2563eb;
    border: 1px solid #93c5fd;
}

.role-4wheel {
    background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
    color: #059669;
    border: 1px solid #6ee7b7;
}

.status-active {
    background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
    color: #059669;
    border: 1px solid #6ee7b7;
}

.status-inactive {
    background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
    color: #dc2626;
    border: 1px solid #fca5a5;
}

/* Action Buttons */
.action-buttons {
    display: flex;
    gap: 0.375rem;
    opacity: 0;
    transition: var(--transition-fast);
}

.table-row:hover .action-buttons {
    opacity: 1;
}

.action-btn {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    border-radius: var(--radius-sm);
    border: none;
    cursor: pointer;
    transition: var(--transition-fast);
    font-size: 0.875rem;
}

.action-edit {
    background: #dbeafe;
    color: #2563eb;
}

.action-edit:hover {
    background: #2563eb;
    color: white;
    transform: scale(1.1);
}

.action-toggle {
    background: #f3f4f6;
    color: #6b7280;
}

.action-toggle:hover {
    background: #6b7280;
    color: white;
    transform: scale(1.1);
}

.action-reset {
    background: #fef3c7;
    color: #d97706;
}

.action-reset:hover {
    background: #d97706;
    color: white;
    transform: scale(1.1);
}

.action-delete {
    background: #fee2e2;
    color: #dc2626;
}

.action-delete:hover {
    background: #dc2626;
    color: white;
    transform: scale(1.1);
}

/* Mobile Views */
.mobile-view {
    display: none;
}

.mobile-card {
    background: white;
    border: 1px solid var(--medium-gray);
    border-radius: var(--radius-lg);
    margin-bottom: 1rem;
    overflow: hidden;
    transition: var(--transition-fast);
}

.mobile-card:hover {
    box-shadow: var(--shadow-md);
}

.mobile-card-header {
    padding: 1rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    position: relative;
}

.mobile-menu-btn {
    position: absolute;
    right: 1rem;
    top: 50%;
    transform: translateY(-50%);
    background: none;
    border: none;
    color: var(--gray);
    font-size: 1.25rem;
    cursor: pointer;
    padding: 0.5rem;
}

.mobile-card-body {
    padding: 0 1rem 1rem;
}

.mobile-info-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px solid #f3f4f6;
}

.mobile-info-row:last-child {
    border-bottom: none;
}

.info-label {
    color: var(--gray);
    font-size: 0.875rem;
}

.info-value {
    font-weight: 500;
    color: var(--dark-gray);
    font-size: 0.875rem;
}

.mobile-actions {
    border-top: 1px solid var(--medium-gray);
    padding: 0.75rem;
    background: #f8f9fa;
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.mobile-action-btn {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1rem;
    border: none;
    border-radius: var(--radius-md);
    cursor: pointer;
    transition: var(--transition-fast);
    font-size: 0.875rem;
    text-align: left;
    width: 100%;
    background: white;
    color: var(--dark-gray);
}

.mobile-action-btn:hover {
    background: #f3f4f6;
}

.mobile-action-btn.action-edit {
    color: #2563eb;
}

.mobile-action-btn.action-toggle {
    color: #6b7280;
}

.mobile-action-btn.action-reset {
    color: #d97706;
}

.mobile-action-btn.action-delete {
    color: #dc2626;
}

/* Empty States */
.empty-state {
    padding: 4rem 2rem;
    text-align: center;
}

.empty-content {
    display: flex;
    flex-direction: column;
    align-items: center;
}

.empty-state-mobile {
    padding: 3rem 1rem;
    text-align: center;
    color: var(--gray);
}

.empty-state-mobile i {
    font-size: 3rem;
    color: #e5e7eb;
    margin-bottom: 1rem;
}

.empty-state-mobile h3 {
    font-size: 1.25rem;
    color: var(--dark-gray);
    margin-bottom: 0.5rem;
}

.empty-state-mobile p {
    font-size: 0.875rem;
    margin-bottom: 1rem;
}

/* Pagination */
.modern-card-footer {
    padding: 1rem 1.5rem;
    border-top: 1px solid var(--medium-gray);
    background: #f8fafc;
}

.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

.pagination-info {
    color: var(--gray);
    font-size: 0.875rem;
}

.pagination-wrapper .pagination {
    display: flex;
    gap: 0.25rem;
    flex-wrap: wrap;
}

.pagination-wrapper .page-link {
    padding: 0.375rem 0.625rem;
    border: 1px solid var(--medium-gray);
    border-radius: var(--radius-sm);
    color: var(--dark-gray);
    text-decoration: none;
    transition: var(--transition-fast);
    font-size: 0.875rem;
}

.pagination-wrapper .page-link:hover {
    background: var(--primary-orange);
    color: white;
    border-color: var(--primary-orange);
}

.pagination-wrapper .page-item.active .page-link {
    background: var(--primary-orange);
    color: white;
    border-color: var(--primary-orange);
}

/* Utility Classes */
.d-mobile-none {
    display: inline;
}

.d-desktop-none {
    display: none;
}

/* Responsive Breakpoints */
@media (max-width: 1280px) {
    .content-container {
        padding: 1.25rem;
    }
    
    .dashboard-grid {
        grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    }
}

@media (max-width: 1024px) {
    .content-container {
        padding: 1rem;
    }
    
    .filter-grid {
        grid-template-columns: 1fr 1fr;
    }
    
    .modern-table {
        font-size: 0.875rem;
    }
    
    .modern-table th,
    .modern-table td {
        padding: 0.75rem;
    }
}

@media (max-width: 768px) {
    /* Content */
    .content-container {
        padding: 0.75rem;
    }
    
    /* Header */
    .page-header {
        padding: 1rem;
    }
    
    .page-header-content {
        flex-direction: column;
        align-items: stretch;
    }
    
    .page-title {
        font-size: 1.25rem;
    }
    
    .page-header-actions {
        flex-direction: row;
        width: 100%;
    }
    
    .page-header-actions .btn {
        flex: 1;
    }
    
    /* Filter Toggle */
    .mobile-filter-toggle {
        display: block;
    }
    
    #filterBody {
        display: none;
    }
    
    #filterBody.show {
        display: block;
    }
    
    .filter-grid {
        grid-template-columns: 1fr;
    }
    
    /* Bulk Actions */
    .bulk-actions-content {
        flex-direction: column;
        align-items: stretch;
    }
    
    .bulk-buttons {
        justify-content: stretch;
    }
    
    .bulk-buttons .btn {
        flex: 1;
    }
    
    /* Hide Desktop Table */
    .desktop-view {
        display: none;
    }
    
    /* Show Mobile Cards */
    .mobile-view {
        display: block;
        padding: 1rem;
    }
    
    /* Dashboard Grid */
    .dashboard-grid {
        grid-template-columns: 1fr;
        gap: 0.75rem;
    }
    
    /* Cards */
    .modern-card {
        margin-bottom: 1rem;
        border-radius: var(--radius-md);
    }
    
    .modern-card-header {
        padding: 0.875rem 1rem;
    }
    
    .modern-card-title {
        font-size: 1rem;
    }
    
    /* Button Text */
    .btn-text {
        display: none;
    }
    
    .btn-sm-full {
        width: 100%;
    }
    
    /* Pagination */
    .pagination-container {
        flex-direction: column;
        align-items: center;
    }
    
    .pagination-info {
        text-align: center;
    }
    
    /* Utility Classes */
    .d-mobile-none {
        display: none;
    }
    
    .d-desktop-none {
        display: inline;
    }
}

@media (max-width: 480px) {
    .content-container {
        padding: 0.5rem;
    }
    
    .page-header {
        border-radius: var(--radius-md);
    }
    
    .modern-card {
        border-radius: var(--radius-md);
    }
    
    .dashboard-card {
        padding: 1.25rem;
    }
    
    .dashboard-card-num {
        font-size: 1.75rem;
    }
    
    .btn {
        padding: 0.5rem 1rem;
        font-size: 0.875rem;
    }
    
    .btn-sm {
        padding: 0.375rem 0.75rem;
        font-size: 0.8rem;
    }
}

/* Print Styles */
@media print {
    .page-header-actions,
    .mobile-filter-toggle,
    .filter-section,
    .bulk-actions-card,
    .action-buttons,
    .mobile-menu-btn,
    .mobile-actions,
    .pagination-container {
        display: none !important;
    }
    
    .desktop-view {
        display: block !important;
    }
    
    .mobile-view {
        display: none !important;
    }
    
    .action-buttons {
        opacity: 0 !important;
    }
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
// Enhanced functionality
document.addEventListener('DOMContentLoaded', function() {
    initializeCheckboxes();
    initializeTableInteractions();
    initializeResponsive();
});

// Initialize responsive features
function initializeResponsive() {
    // Handle window resize
    let resizeTimer;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(function() {
            adjustLayout();
        }, 250);
    });
    
    // Initial layout adjustment
    adjustLayout();
}

// Adjust layout based on screen size
function adjustLayout() {
    const width = window.innerWidth;
    
    if (width <= 768) {
        // Mobile adjustments
        document.querySelectorAll('.btn-text').forEach(el => {
            el.style.display = 'none';
        });
    } else {
        // Desktop adjustments
        document.querySelectorAll('.btn-text').forEach(el => {
            el.style.display = 'inline';
        });
    }
}

// Toggle filter visibility on mobile
function toggleFilters() {
    const filterBody = document.getElementById('filterBody');
    filterBody.classList.toggle('show');
    
    // Animate
    if (filterBody.classList.contains('show')) {
        filterBody.style.maxHeight = filterBody.scrollHeight + 'px';
    } else {
        filterBody.style.maxHeight = '0';
    }
}

// Toggle mobile menu for user actions
function toggleMobileMenu(userId) {
    const menu = document.getElementById('mobile-actions-' + userId);
    const allMenus = document.querySelectorAll('.mobile-actions');
    
    // Close all other menus
    allMenus.forEach(m => {
        if (m.id !== 'mobile-actions-' + userId) {
            m.style.display = 'none';
        }
    });
    
    // Toggle current menu
    if (menu.style.display === 'none' || !menu.style.display) {
        menu.style.display = 'flex';
        // Add slide animation
        menu.style.animation = 'slideDown 0.3s ease-out';
    } else {
        menu.style.display = 'none';
    }
}

// Close mobile menus when clicking outside
document.addEventListener('click', function(e) {
    if (!e.target.closest('.mobile-menu-btn') && !e.target.closest('.mobile-actions')) {
        document.querySelectorAll('.mobile-actions').forEach(menu => {
            menu.style.display = 'none';
        });
    }
});

// Initialize checkboxes
function initializeCheckboxes() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('.user-checkbox');
    
    selectAll?.addEventListener('change', function() {
        checkboxes.forEach(checkbox => {
            checkbox.checked = this.checked;
        });
        updateBulkActions();
    });
    
    checkboxes.forEach(checkbox => {
        checkbox.addEventListener('change', updateBulkActions);
    });
}

// Initialize table interactions
function initializeTableInteractions() {
    // Desktop table rows
    const rows = document.querySelectorAll('.table-row');
    rows.forEach(row => {
        row.addEventListener('click', function(e) {
            if (e.target.closest('.action-btn, .modern-checkbox')) {
                return;
            }
            
            const checkbox = this.querySelector('.user-checkbox');
            if (checkbox) {
                checkbox.checked = !checkbox.checked;
                updateBulkActions();
            }
        });
    });
    
    // Mobile cards
    const cards = document.querySelectorAll('.mobile-card');
    cards.forEach(card => {
        card.addEventListener('click', function(e) {
            if (e.target.closest('.mobile-menu-btn, .mobile-actions, .modern-checkbox')) {
                return;
            }
            
            const checkbox = this.querySelector('.user-checkbox');
            if (checkbox) {
                checkbox.checked = !checkbox.checked;
                updateBulkActions();
            }
        });
    });
}

// Update bulk actions visibility
function updateBulkActions() {
    const selectedCheckboxes = document.querySelectorAll('.user-checkbox:checked');
    const bulkActions = document.getElementById('bulkActions');
    const selectedCount = document.getElementById('selectedCount');
    const selectAll = document.getElementById('selectAll');
    
    if (selectedCheckboxes.length > 0) {
        bulkActions.style.display = 'block';
        selectedCount.textContent = selectedCheckboxes.length;
        
        // Smooth animation
        if (!bulkActions.classList.contains('show')) {
            bulkActions.style.opacity = '0';
            bulkActions.style.transform = 'translateY(-10px)';
            bulkActions.classList.add('show');
            
            setTimeout(() => {
                bulkActions.style.transition = 'all 0.3s ease';
                bulkActions.style.opacity = '1';
                bulkActions.style.transform = 'translateY(0)';
            }, 50);
        }
    } else {
        bulkActions.style.display = 'none';
        bulkActions.classList.remove('show');
    }
    
    // Update select all checkbox state
    const allCheckboxes = document.querySelectorAll('.user-checkbox');
    if (selectAll) {
        selectAll.checked = allCheckboxes.length > 0 && selectedCheckboxes.length === allCheckboxes.length;
        selectAll.indeterminate = selectedCheckboxes.length > 0 && selectedCheckboxes.length < allCheckboxes.length;
    }
}

// Clear selection
function clearSelection() {
    document.querySelectorAll('.user-checkbox').forEach(checkbox => {
        checkbox.checked = false;
    });
    document.getElementById('selectAll').checked = false;
    updateBulkActions();
    
    // Show toast
    showToast('info', 'ยกเลิกการเลือกแล้ว');
}

// Show toast notification
function showToast(icon, title) {
    const Toast = Swal.mixin({
        toast: true,
        position: 'top-end',
        showConfirmButton: false,
        timer: 3000,
        timerProgressBar: true,
        didOpen: (toast) => {
            toast.addEventListener('mouseenter', Swal.stopTimer)
            toast.addEventListener('mouseleave', Swal.resumeTimer)
        }
    });
    
    Toast.fire({
        icon: icon,
        title: title
    });
}

// Filter functions
function filterByStatus(status) {
    const url = new URL(window.location);
    url.searchParams.set('status', status);
    window.location.href = url.toString();
}

function filterByRole(role) {
    const url = new URL(window.location);
    url.searchParams.set('role', role);
    window.location.href = url.toString();
}

// Export users
function exportUsers() {
    const params = new URLSearchParams(window.location.search);
    const exportUrl = `{{ route('admin.users.export') }}?${params.toString()}`;
    
    // Show loading
    Swal.fire({
        title: 'กำลังเตรียมข้อมูล',
        text: 'กรุณารอสักครู่...',
        allowOutsideClick: false,
        didOpen: () => {
            Swal.showLoading();
        }
    });
    
    // Create temporary link for download
    const link = document.createElement('a');
    link.href = exportUrl;
    link.download = `users_${new Date().toISOString().split('T')[0]}.csv`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    // Close loading and show success
    setTimeout(() => {
        Swal.close();
        showToast('success', 'ส่งออกข้อมูลสำเร็จ!');
    }, 1000);
}

// Bulk actions
function bulkAction(action) {
    const selectedIds = Array.from(document.querySelectorAll('.user-checkbox:checked')).map(cb => cb.value);
    
    if (selectedIds.length === 0) {
        Swal.fire({
            icon: 'warning',
            title: 'ยังไม่ได้เลือกผู้ใช้',
            text: 'กรุณาเลือกผู้ใช้อย่างน้อย 1 คน',
            confirmButtonColor: '#FF6B35'
        });
        return;
    }

    const actionConfig = {
        activate: {
            title: 'เปิดใช้งานผู้ใช้',
            text: `คุณต้องการเปิดใช้งานผู้ใช้ ${selectedIds.length} คนที่เลือกหรือไม่?`,
            icon: 'question',
            confirmButtonText: 'เปิดใช้งาน',
            color: '#22C55E'
        },
        deactivate: {
            title: 'ปิดใช้งานผู้ใช้',
            text: `คุณต้องการปิดใช้งานผู้ใช้ ${selectedIds.length} คนที่เลือกหรือไม่?`,
            icon: 'warning',
            confirmButtonText: 'ปิดใช้งาน',
            color: '#F59E0B'
        }
    };
    
    const config = actionConfig[action];
    
    Swal.fire({
        title: config.title,
        text: config.text,
        icon: config.icon,
        showCancelButton: true,
        confirmButtonColor: config.color,
        cancelButtonColor: '#6B7280',
        confirmButtonText: config.confirmButtonText,
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`{{ route('admin.users.bulk') }}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    action: action,
                    user_ids: selectedIds
                })
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(data => {
                if (!data.success) {
                    throw new Error(data.message || 'เกิดข้อผิดพลาด');
                }
                return data;
            })
            .catch(error => {
                Swal.showValidationMessage(`เกิดข้อผิดพลาด: ${error.message}`);
            });
        },
        allowOutsideClick: () => !Swal.isLoading()
    }).then((result) => {
        if (result.isConfirmed && result.value) {
            Swal.fire({
                icon: 'success',
                title: 'สำเร็จ!',
                text: result.value.message,
                confirmButtonColor: '#FF6B35'
            }).then(() => {
                location.reload();
            });
        }
    });
}

// Individual user actions
function toggleUserStatus(userId) {
    Swal.fire({
        title: 'ยืนยันการเปลี่ยนสถานะ',
        text: 'คุณต้องการเปลี่ยนสถานะการใช้งานของผู้ใช้นี้หรือไม่?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#FF6B35',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'ยืนยัน',
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`{{ route('admin.users.index') }}/${userId}/toggle`, {
                method: 'PATCH',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (!data.success) {
                    throw new Error(data.message || 'เกิดข้อผิดพลาด');
                }
                return data;
            })
            .catch(error => {
                Swal.showValidationMessage(`เกิดข้อผิดพลาด: ${error.message}`);
            });
        }
    }).then((result) => {
        if (result.isConfirmed && result.value) {
            Swal.fire({
                icon: 'success',
                title: 'สำเร็จ!',
                text: result.value.message,
                confirmButtonColor: '#FF6B35'
            }).then(() => {
                location.reload();
            });
        }
    });
}

function resetPassword(userId) {
    Swal.fire({
        title: 'ยืนยันการรีเซ็ตรหัสผ่าน',
        text: 'คุณต้องการรีเซ็ตรหัสผ่านของผู้ใช้นี้หรือไม่?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#F59E0B',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'รีเซ็ตรหัสผ่าน',
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`{{ route('admin.users.index') }}/${userId}/reset-password`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (!data.success) {
                    throw new Error(data.message || 'เกิดข้อผิดพลาด');
                }
                return data;
            })
            .catch(error => {
                Swal.showValidationMessage(`เกิดข้อผิดพลาด: ${error.message}`);
            });
        }
    }).then((result) => {
        if (result.isConfirmed && result.value) {
            Swal.fire({
                icon: 'success',
                title: 'รีเซ็ตรหัสผ่านสำเร็จ!',
                html: `
                    <div class="text-left bg-gray-100 p-4 rounded-lg mt-4">
                        <div class="font-semibold mb-2">รหัสผ่านใหม่:</div>
                        <div class="font-mono text-lg bg-white p-2 rounded border-2 border-dashed border-orange-300">
                            ${result.value.new_password}
                        </div>
                        <div class="text-sm text-gray-600 mt-2">
                            กรุณาบันทึกรหัสผ่านนี้และแจ้งให้ผู้ใช้ทราบ
                        </div>
                    </div>
                `,
                confirmButtonColor: '#FF6B35',
                confirmButtonText: 'เข้าใจแล้ว'
            });
        }
    });
}

function deleteUser(userId) {
    Swal.fire({
        title: 'ยืนยันการลบผู้ใช้',
        text: 'คุณต้องการลบผู้ใช้นี้หรือไม่? การดำเนินการนี้ไม่สามารถยกเลิกได้',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#EF4444',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'ลบผู้ใช้',
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`{{ route('admin.users.index') }}/${userId}`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (!data.success) {
                    throw new Error(data.message || 'เกิดข้อผิดพลาด');
                }
                return data;
            })
            .catch(error => {
                Swal.showValidationMessage(`เกิดข้อผิดพลาด: ${error.message}`);
            });
        }
    }).then((result) => {
        if (result.isConfirmed && result.value) {
            Swal.fire({
                icon: 'success',
                title: 'ลบผู้ใช้สำเร็จ!',
                text: result.value.message,
                confirmButtonColor: '#FF6B35'
            }).then(() => {
                location.reload();
            });
        }
    });
}

// Auto-submit form on select changes
document.addEventListener('DOMContentLoaded', function() {
    document.querySelectorAll('select[name="role"], select[name="status"]').forEach(select => {
        select.addEventListener('change', function() {
            this.closest('form').submit();
        });
    });
});

// Enhanced search functionality with debounce
let searchTimeout;
const searchInput = document.querySelector('input[name="search"]');
if (searchInput) {
    searchInput.addEventListener('input', function() {
        clearTimeout(searchTimeout);
        
        // Add visual feedback
        this.style.borderColor = '#FF6B35';
        
        searchTimeout = setTimeout(() => {
            this.style.borderColor = '';
        }, 1000);
    });
}

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    // Ctrl/Cmd + A to select all
    if ((e.ctrlKey || e.metaKey) && e.key === 'a' && !e.target.matches('input, textarea')) {
        e.preventDefault();
        const selectAll = document.getElementById('selectAll');
        if (selectAll) {
            selectAll.checked = true;
            selectAll.dispatchEvent(new Event('change'));
        }
    }
    
    // Escape to clear selection
    if (e.key === 'Escape') {
        clearSelection();
    }
    
    // Delete key for bulk delete (when items are selected)
    if (e.key === 'Delete') {
        const selectedIds = Array.from(document.querySelectorAll('.user-checkbox:checked')).map(cb => cb.value);
        if (selectedIds.length > 0) {
            e.preventDefault();
            bulkAction('deactivate');
        }
    }
});

// Add slide animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideDown {
        from {
            opacity: 0;
            transform: translateY(-10px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
    
    @keyframes fadeIn {
        from {
            opacity: 0;
        }
        to {
            opacity: 1;
        }
    }
`;
document.head.appendChild(style);

// Show success/error messages from session
@if(session('success'))
    document.addEventListener('DOMContentLoaded', function() {
        showToast('success', '{{ session('success') }}');
    });
@endif

@if(session('error'))
    document.addEventListener('DOMContentLoaded', function() {
        Swal.fire({
            icon: 'error',
            title: 'เกิดข้อผิดพลาด!',
            text: '{{ session('error') }}',
            confirmButtonColor: '#FF6B35'
        });
    });
@endif

// Responsive table scroll indicator
function checkTableScroll() {
    const tableWrapper = document.querySelector('.table-responsive');
    if (tableWrapper) {
        const isScrollable = tableWrapper.scrollWidth > tableWrapper.clientWidth;
        if (isScrollable) {
            tableWrapper.classList.add('scrollable');
        } else {
            tableWrapper.classList.remove('scrollable');
        }
    }
}

// Check on load and resize
window.addEventListener('load', checkTableScroll);
window.addEventListener('resize', checkTableScroll);

// Add scroll indicator styles
const scrollStyle = document.createElement('style');
scrollStyle.textContent = `
    .table-responsive.scrollable {
        position: relative;
    }
    
    .table-responsive.scrollable::after {
        content: '';
        position: absolute;
        right: 0;
        top: 0;
        bottom: 0;
        width: 30px;
        background: linear-gradient(to right, transparent, rgba(0,0,0,0.1));
        pointer-events: none;
    }
`;
document.head.appendChild(scrollStyle);
</script>
@endpush