@extends('layouts.app')

@section('title', 'แก้ไขคลังสินค้า - ระบบจัดการค่าคอมมิชชั่น')

@section('content')
<div class="content-container">
    <!-- Enhanced Page Header -->
    <div class="page-header">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="page-title">
                    <i class="fas fa-edit text-blue-500 mr-3"></i>
                    แก้ไขคลังสินค้า
                </h1>
                <p class="page-subtitle">แก้ไขข้อมูล: <span class="text-orange-600 font-semibold">{{ $warehouse->code }}</span> - {{ $warehouse->name }}</p>
            </div>
            <div class="flex space-x-3">
                <a href="{{ route('admin.commission-rates.index', ['warehouse' => $warehouse->code]) }}" 
                   class="btn btn-info">
                    <i class="fas fa-percentage"></i>จัดการอัตราค่าคอมมิชชั่น
                </a>
                <a href="{{ route('admin.warehouses.index') }}" class="btn btn-secondary">
                    <i class="fas fa-arrow-left"></i>กลับไปรายการ
                </a>
            </div>
        </div>
    </div>

    <!-- Stats Overview -->
    <div class="stats-overview">
        <div class="stat-item">
            <div class="stat-icon stat-icon-users">
                <i class="fas fa-users"></i>
            </div>
            <div class="stat-content">
                <div class="stat-number">{{ $warehouse->users->count() }}</div>
                <div class="stat-label">ผู้ใช้ทั้งหมด</div>
                <div class="stat-detail">{{ $warehouse->users->where('is_active', true)->count() }} ใช้งานอยู่</div>
            </div>
        </div>
        
        <div class="stat-item">
            <div class="stat-icon stat-icon-rates">
                <i class="fas fa-percentage"></i>
            </div>
            <div class="stat-content">
                <div class="stat-number">{{ $warehouse->commissionRates->count() }}</div>
                <div class="stat-label">อัตราค่าคอมมิชชั่น</div>
                <div class="stat-detail">{{ $warehouse->commissionRates->where('is_active', true)->count() }} ใช้งานอยู่</div>
            </div>
        </div>
        
        <div class="stat-item">
            <div class="stat-icon stat-icon-status">
                <i class="fas fa-{{ $warehouse->is_active ? 'check-circle' : 'times-circle' }}"></i>
            </div>
            <div class="stat-content">
                <div class="stat-number">{{ $warehouse->is_active ? 'เปิด' : 'ปิด' }}</div>
                <div class="stat-label">สถานะ</div>
                <div class="stat-detail">{{ $warehouse->is_active ? 'ใช้งานอยู่' : 'ปิดใช้งาน' }}</div>
            </div>
        </div>
        
        <div class="stat-item">
            <div class="stat-icon stat-icon-time">
                <i class="fas fa-calendar-alt"></i>
            </div>
            <div class="stat-content">
                <div class="stat-number">{{ $warehouse->created_at ? $warehouse->created_at->diffForHumans() : '-' }}</div>
                <div class="stat-label">สร้างเมื่อ</div>
                <div class="stat-detail">{{ $warehouse->created_at ? $warehouse->created_at->format('d/m/Y H:i') : '-' }}</div>
            </div>
        </div>
    </div>

    <!-- Enhanced Form Container -->
    <div class="modern-card">
        <div class="modern-card-header">
            <h3 class="modern-card-title">
                <i class="fas fa-warehouse text-green-500"></i>
                ข้อมูลคลังสินค้า
            </h3>
        </div>
        
        <div class="modern-card-body">
            <form method="POST" action="{{ route('admin.warehouses.update', $warehouse) }}" id="warehouseForm">
                @csrf
                @method('PUT')
                
                <div class="form-grid">
                    <!-- รหัสคลัง -->
                    <div class="form-group">
                        <label for="code" class="form-label required">รหัสคลัง</label>
                        <div class="input-with-icon">
                            <i class="fas fa-barcode input-icon"></i>
                            <input type="text" id="code" name="code" 
                                   value="{{ old('code', $warehouse->code) }}" 
                                   class="form-control @error('code') border-red-500 @enderror" 
                                   placeholder="เช่น WH001, BKK01" required maxlength="10">
                        </div>
                        @error('code')
                            <p class="error-message">{{ $message }}</p>
                        @enderror
                        <p class="help-text">รหัสคลังต้องไม่ซ้ำกัน และมีได้สูงสุด 10 ตัวอักษร</p>
                    </div>

                    <!-- ชื่อคลัง -->
                    <div class="form-group">
                        <label for="name" class="form-label required">ชื่อคลัง</label>
                        <div class="input-with-icon">
                            <i class="fas fa-warehouse input-icon"></i>
                            <input type="text" id="name" name="name" 
                                   value="{{ old('name', $warehouse->name) }}" 
                                   class="form-control @error('name') border-red-500 @enderror" 
                                   placeholder="เช่น คลังสินค้ากรุงเทพ" required>
                        </div>
                        @error('name')
                            <p class="error-message">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- ที่อยู่ -->
                    <div class="form-group full-width">
                        <label for="location" class="form-label">ที่อยู่</label>
                        <div class="textarea-wrapper">
                            <i class="fas fa-map-marker-alt textarea-icon"></i>
                            <textarea id="location" name="location" rows="3" 
                                      class="form-control @error('location') border-red-500 @enderror" 
                                      placeholder="ระบุที่อยู่ของคลังสินค้า">{{ old('location', $warehouse->location) }}</textarea>
                        </div>
                        @error('location')
                            <p class="error-message">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- สถานะ -->
                    <div class="form-group">
                        <label class="form-label">สถานะ</label>
                        <div class="radio-group">
                            <label class="radio-option {{ old('is_active', $warehouse->is_active) ? 'active' : '' }}">
                                <input type="radio" name="is_active" value="1" 
                                       {{ old('is_active', $warehouse->is_active) ? 'checked' : '' }}>
                                <span class="radio-custom"></span>
                                <span class="radio-label">
                                    <i class="fas fa-check-circle text-green-500"></i>
                                    ใช้งานอยู่
                                </span>
                            </label>
                            <label class="radio-option {{ !old('is_active', $warehouse->is_active) ? 'active' : '' }}">
                                <input type="radio" name="is_active" value="0" 
                                       {{ !old('is_active', $warehouse->is_active) ? 'checked' : '' }}>
                                <span class="radio-custom"></span>
                                <span class="radio-label">
                                    <i class="fas fa-times-circle text-red-500"></i>
                                    ปิดใช้งาน
                                </span>
                            </label>
                        </div>
                    </div>

                    <!-- ข้อมูลระบบ -->
                    <div class="form-group system-info">
                        <label class="form-label">ข้อมูลระบบ</label>
                        <div class="system-card">
                            <div class="system-item">
                                <div class="system-label">สร้างเมื่อ</div>
                                <div class="system-value">
                                    {{ $warehouse->created_at ? $warehouse->created_at->format('d/m/Y H:i:s') : '-' }}
                                </div>
                            </div>
                            <div class="system-item">
                                <div class="system-label">อัปเดตล่าสุด</div>
                                <div class="system-value">
                                    {{ $warehouse->updated_at ? $warehouse->updated_at->format('d/m/Y H:i:s') : '-' }}
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Enhanced Submit Section -->
                <div class="form-actions">
                    <div class="actions-left">
                        @if($warehouse->users->count() === 0)
                        <button type="button" onclick="deleteWarehouse()" class="btn btn-danger">
                            <i class="fas fa-trash"></i>ลบคลัง
                        </button>
                        @else
                        <div class="warning-note">
                            <i class="fas fa-exclamation-triangle"></i>
                            <span>ไม่สามารถลบคลังที่มีผู้ใช้งานได้</span>
                        </div>
                        @endif
                    </div>
                    
                    <div class="actions-right">
                        <a href="{{ route('admin.warehouses.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times"></i>ยกเลิก
                        </a>

                        <button type="submit" class="btn btn-primary" id="submitBtn">
                            <i class="fas fa-save"></i>บันทึกการเปลี่ยนแปลง
                            <span class="btn-loading" style="display: none;">
                                <i class="fas fa-spinner fa-spin"></i>
                            </span>
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Users & Rates Overview -->
    @if($warehouse->users->count() > 0 || $warehouse->commissionRates->count() > 0)
    <div class="overview-section">
        <div class="overview-grid">
            <!-- Users Overview -->
            @if($warehouse->users->count() > 0)
            <div class="modern-card">
                <div class="modern-card-header">
                    <h3 class="modern-card-title">
                        <i class="fas fa-users text-purple-500"></i>
                        ผู้ใช้งานในคลัง ({{ $warehouse->users->count() }} คน)
                    </h3>
                </div>
                <div class="modern-card-body">
                    <div class="users-list">
                        @foreach($warehouse->users->take(5) as $user)
                        <div class="user-item">
                            <div class="user-avatar">
                                {{ strtoupper(substr($user->name, 0, 2)) }}
                            </div>
                            <div class="user-details">
                                <div class="user-name">{{ $user->name }}</div>
                                <div class="user-role">{{ $user->role }}</div>
                            </div>
                            <div class="user-status">
                                <span class="status-badge status-{{ $user->is_active ? 'active' : 'inactive' }}">
                                    {{ $user->is_active ? 'ใช้งานอยู่' : 'ปิดใช้งาน' }}
                                </span>
                            </div>
                        </div>
                        @endforeach
                        
                        @if($warehouse->users->count() > 5)
                        <div class="show-more">
                            <a href="{{ route('admin.users.index', ['warehouse' => $warehouse->code]) }}" class="btn btn-outline">
                                ดูทั้งหมด {{ $warehouse->users->count() }} คน
                            </a>
                        </div>
                        @endif
                    </div>
                </div>
            </div>
            @endif

            <!-- Commission Rates Overview -->
            @if($warehouse->commissionRates->count() > 0)
            <div class="modern-card">
                <div class="modern-card-header">
                    <h3 class="modern-card-title">
                        <i class="fas fa-percentage text-orange-500"></i>
                        อัตราค่าคอมมิชชั่น ({{ $warehouse->commissionRates->count() }} อัตรา)
                    </h3>
                </div>
                <div class="modern-card-body">
                    <div class="rates-list">
                        @foreach($warehouse->commissionRates->take(5) as $rate)
                        <div class="rate-item">
                            <div class="rate-info">
                                <div class="rate-vehicle">
                                    <i class="fas fa-{{ $rate->vehicle_type === '2wheel' ? 'motorcycle' : 'truck' }}"></i>
                                    {{ $rate->vehicle_type === '2wheel' ? 'รถ 2 ล้อ' : 'รถ 4 ล้อ' }}
                                </div>
                                <div class="rate-range">{{ $rate->min_amount }}-{{ $rate->max_amount }} บาท</div>
                            </div>
                            <div class="rate-percentage">{{ $rate->commission_rate }}%</div>
                            <div class="rate-status">
                                <span class="status-badge status-{{ $rate->is_active ? 'active' : 'inactive' }}">
                                    {{ $rate->is_active ? 'ใช้งานอยู่' : 'ปิดใช้งาน' }}
                                </span>
                            </div>
                        </div>
                        @endforeach
                        
                        @if($warehouse->commissionRates->count() > 5)
                        <div class="show-more">
                            <a href="{{ route('admin.commission-rates.index', ['warehouse' => $warehouse->code]) }}" class="btn btn-outline">
                                ดูทั้งหมด {{ $warehouse->commissionRates->count() }} อัตรา
                            </a>
                        </div>
                        @endif
                    </div>
                </div>
            </div>
            @endif
        </div>
    </div>
    @endif
</div>

<!-- Enhanced Styles -->
<style>
/* Fix Horizontal Scrollbar */
* {
    box-sizing: border-box;
}

.content-container {
    max-width: 100vw;
    overflow-x: hidden;
    padding-left: 1rem;
    padding-right: 1rem;
    box-sizing: border-box;
}

/* Stats Overview */
.stats-overview {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-item {
    background: white;
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    border: 1px solid var(--medium-gray);
    display: flex;
    align-items: center;
    gap: 1rem;
    transition: var(--transition-fast);
}

.stat-item:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.stat-icon {
    width: 60px;
    height: 60px;
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.8rem;
    color: white;
    flex-shrink: 0;
}

.stat-icon-users {
    background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);
}

.stat-icon-rates {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
}

.stat-icon-status {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
}

.stat-icon-time {
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
}

.stat-content {
    flex: 1;
}

.stat-number {
    font-size: 1.8rem;
    font-weight: 700;
    color: var(--dark-gray);
    margin-bottom: 0.25rem;
}

.stat-label {
    font-size: 1rem;
    font-weight: 600;
    color: var(--gray);
    margin-bottom: 0.25rem;
}

.stat-detail {
    font-size: 0.875rem;
    color: var(--gray);
}

/* Form Grid Layout */
.form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
    margin-bottom: 2rem;
}

.form-group.full-width {
    grid-column: 1 / -1;
}

.form-group.system-info {
    grid-column: 1 / -1;
}

/* Enhanced Form Elements */
.form-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.form-label {
    font-weight: 600;
    color: var(--dark-gray);
    font-size: 0.95rem;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.form-label.required::after {
    content: '*';
    color: #ef4444;
    font-weight: bold;
}

.input-with-icon {
    position: relative;
    display: flex;
    align-items: center;
}

.input-icon {
    position: absolute;
    left: 1rem;
    color: var(--gray);
    font-size: 1rem;
    z-index: 1;
}

.form-control {
    width: 100%;
    padding: 1rem 1.25rem 1rem 3rem;
    border: 2px solid var(--medium-gray);
    border-radius: var(--radius-lg);
    font-family: inherit;
    font-size: 1rem;
    transition: var(--transition-fast);
    background: var(--white);
    line-height: 1.5;
}

.form-control:focus {
    outline: none;
    border-color: var(--primary-orange);
    box-shadow: 0 0 0 4px rgba(255, 107, 53, 0.1);
    transform: translateY(-1px);
}

.form-control::placeholder {
    color: var(--gray);
    opacity: 0.7;
}

/* Textarea Styling */
.textarea-wrapper {
    position: relative;
}

.textarea-icon {
    position: absolute;
    top: 1rem;
    left: 1rem;
    color: var(--gray);
    font-size: 1rem;
    z-index: 1;
}

textarea.form-control {
    padding-left: 3rem;
    padding-top: 1rem;
    resize: vertical;
    min-height: 100px;
}

/* Radio Group Styling */
.radio-group {
    display: flex;
    gap: 1.5rem;
    margin-top: 0.5rem;
}

.radio-option {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    cursor: pointer;
    padding: 0.75rem 1rem;
    border: 2px solid var(--medium-gray);
    border-radius: var(--radius-md);
    transition: var(--transition-fast);
    flex: 1;
    position: relative;
}

.radio-option:hover {
    border-color: var(--primary-orange);
    background: rgba(255, 107, 53, 0.02);
}

.radio-option.active {
    border-color: var(--primary-orange);
    background: rgba(255, 107, 53, 0.05);
}

.radio-option input[type="radio"] {
    display: none;
}

.radio-custom {
    width: 20px;
    height: 20px;
    border: 2px solid var(--medium-gray);
    border-radius: 50%;
    position: relative;
    transition: var(--transition-fast);
}

.radio-option input[type="radio"]:checked + .radio-custom {
    border-color: var(--primary-orange);
    background: var(--primary-orange);
}

.radio-option input[type="radio"]:checked + .radio-custom::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 8px;
    height: 8px;
    background: white;
    border-radius: 50%;
}

.radio-label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 500;
    color: var(--dark-gray);
}

/* System Info Card */
.system-card {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border: 1px solid #e2e8f0;
    border-radius: var(--radius-md);
    padding: 1rem;
    display: flex;
    gap: 2rem;
}

.system-item {
    flex: 1;
}

.system-label {
    font-size: 0.875rem;
    font-weight: 500;
    color: var(--gray);
    margin-bottom: 0.25rem;
}

.system-value {
    font-size: 0.95rem;
    font-weight: 600;
    color: var(--dark-gray);
}

/* Error Message Styling */
.error-message {
    color: #ef4444;
    font-size: 0.875rem;
    margin-top: 0.25rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.error-message::before {
    content: '⚠';
    color: #ef4444;
}

/* Help Text Styling */
.help-text {
    color: var(--gray);
    font-size: 0.875rem;
    margin-top: 0.25rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.help-text::before {
    content: 'ℹ';
    color: #3b82f6;
    font-weight: bold;
}

/* Form Actions */
.form-actions {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding-top: 2rem;
    border-top: 2px solid var(--medium-gray);
    margin-top: 2rem;
}

.actions-left {
    flex: 1;
}

.actions-right {
    display: flex;
    gap: 1rem;
}

.warning-note {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #f59e0b;
    font-size: 0.875rem;
    font-weight: 500;
}

/* Button Enhancements */
.btn {
    position: relative;
    overflow: hidden;
}

.btn-loading {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
}

.btn.loading .btn-loading {
    display: block;
}

.btn.loading i:not(.btn-loading i) {
    opacity: 0;
}

.btn-secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: white;
    border: 2px solid transparent;
}

.btn-secondary:hover {
    background: linear-gradient(135deg, #4b5563 0%, #374151 100%);
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.btn-danger {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
    border: 2px solid transparent;
}

.btn-danger:hover {
    background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%);
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.btn-info {
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
    color: white;
    border: 2px solid transparent;
}

.btn-info:hover {
    background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.btn-outline {
    background: transparent;
    color: var(--primary-orange);
    border: 2px solid var(--primary-orange);
}

.btn-outline:hover {
    background: var(--primary-orange);
    color: white;
    transform: translateY(-1px);
}

/* Overview Section */
.overview-section {
    margin-top: 2rem;
}

.overview-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
}

/* Users List */
.users-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.user-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem;
    border: 1px solid var(--medium-gray);
    border-radius: var(--radius-md);
    transition: var(--transition-fast);
}

.user-item:hover {
    border-color: var(--primary-orange);
    background: rgba(255, 107, 53, 0.02);
}

.user-avatar {
    width: 40px;
    height: 40px;
    border-radius: var(--radius-md);
    background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
    font-size: 1rem;
    flex-shrink: 0;
}

.user-details {
    flex: 1;
}

.user-name {
    font-weight: 600;
    color: var(--dark-gray);
    margin-bottom: 0.25rem;
}

.user-role {
    font-size: 0.875rem;
    color: var(--gray);
}

.user-status {
    flex-shrink: 0;
}

/* Rates List */
.rates-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.rate-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem;
    border: 1px solid var(--medium-gray);
    border-radius: var(--radius-md);
    transition: var(--transition-fast);
}

.rate-item:hover {
    border-color: var(--primary-orange);
    background: rgba(255, 107, 53, 0.02);
}

.rate-info {
    flex: 1;
}

.rate-vehicle {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 600;
    color: var(--dark-gray);
    margin-bottom: 0.25rem;
}

.rate-range {
    font-size: 0.875rem;
    color: var(--gray);
}

.rate-percentage {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--primary-orange);
    flex-shrink: 0;
    margin-right: 1rem;
}

.rate-status {
    flex-shrink: 0;
}

/* Status Badge */
.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 0.25rem 0.75rem;
    border-radius: var(--radius-full);
    font-size: 0.75rem;
    font-weight: 600;
    white-space: nowrap;
}

.status-active {
    background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
    color: #059669;
}

.status-inactive {
    background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
    color: #dc2626;
}

/* Show More */
.show-more {
    padding-top: 1rem;
    border-top: 1px solid var(--medium-gray);
    text-align: center;
}

/* Responsive Design */
@media (max-width: 1024px) {
    .form-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }
    
    .form-actions {
        flex-direction: column;
        gap: 1rem;
        align-items: stretch;
    }
    
    .actions-right {
        width: 100%;
        justify-content: center;
    }
    
    .overview-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }
    
    .stats-overview {
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 1rem;
    }
}

@media (max-width: 768px) {
    .content-container {
        padding-left: 0.5rem;
        padding-right: 0.5rem;
    }
    
    .radio-group {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .radio-option {
        justify-content: flex-start;
    }
    
    .system-card {
        flex-direction: column;
        gap: 1rem;
    }
    
    .form-control {
        padding: 0.875rem 1rem 0.875rem 2.75rem;
    }
    
    .input-icon {
        left: 0.875rem;
        font-size: 0.9rem;
    }
    
    .stats-overview {
        grid-template-columns: 1fr;
    }
    
    .stat-item {
        padding: 1rem;
    }
    
    .stat-icon {
        width: 50px;
        height: 50px;
        font-size: 1.5rem;
    }
    
    .stat-number {
        font-size: 1.5rem;
    }
    
    .page-header .flex {
        flex-direction: column;
        align-items: stretch;
        gap: 1rem;
    }
}

@media (max-width: 480px) {
    .content-container {
        padding-left: 0.25rem;
        padding-right: 0.25rem;
    }
    
    .form-grid {
        gap: 1rem;
    }
    
    .actions-right {
        flex-direction: column;
    }
    
    .btn {
        width: 100%;
        justify-content: center;
    }
    
    .user-item,
    .rate-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.75rem;
    }
    
    .user-status,
    .rate-status {
        align-self: flex-end;
    }
}
</style>

<!-- Enhanced JavaScript -->
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
// Enhanced functionality
document.addEventListener('DOMContentLoaded', function() {
    initializeForm();
    setupValidation();
    setupInteractions();
});

// Initialize form
function initializeForm() {
    // Auto format warehouse code to uppercase
    const codeInput = document.getElementById('code');
    if (codeInput) {
        codeInput.addEventListener('input', function() {
            this.value = this.value.toUpperCase().replace(/[^A-Z0-9]/g, '');
            validateField(this);
        });
    }
    
    // Radio button interactions
    const radioOptions = document.querySelectorAll('.radio-option');
    radioOptions.forEach(option => {
        option.addEventListener('click', function() {
            const radio = this.querySelector('input[type="radio"]');
            if (radio) {
                radio.checked = true;
                updateRadioStates();
            }
        });
    });
    
    // Initial radio state
    updateRadioStates();
}

// Setup form validation
function setupValidation() {
    const form = document.getElementById('warehouseForm');
    const submitBtn = document.getElementById('submitBtn');
    
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        
        if (validateForm()) {
            submitForm();
        }
    });
    
    // Real-time validation
    const inputs = form.querySelectorAll('input[required], textarea[required]');
    inputs.forEach(input => {
        input.addEventListener('blur', () => validateField(input));
        input.addEventListener('input', () => clearFieldError(input));
    });
}

// Setup interactions
function setupInteractions() {
    // Form field focus effects
    const formControls = document.querySelectorAll('.form-control');
    formControls.forEach(control => {
        control.addEventListener('focus', function() {
            this.parentElement.classList.add('focused');
        });
        
        control.addEventListener('blur', function() {
            this.parentElement.classList.remove('focused');
        });
    });
}

// Validate individual field
function validateField(field) {
    const value = field.value.trim();
    let isValid = true;
    let message = '';
    
    // Required field validation
    if (field.hasAttribute('required') && !value) {
        isValid = false;
        message = 'กรุณากรอกข้อมูลในช่องนี้';
    }
    
    // Specific validations
    if (field.id === 'code' && value) {
        if (value.length < 2) {
            isValid = false;
            message = 'รหัสคลังต้องมีอย่างน้อย 2 ตัวอักษร';
        } else if (value.length > 10) {
            isValid = false;
            message = 'รหัสคลังต้องมีไม่เกิน 10 ตัวอักษร';
        }
    }
    
    if (field.id === 'name' && value && value.length < 3) {
        isValid = false;
        message = 'ชื่อคลังต้องมีอย่างน้อย 3 ตัวอักษร';
    }
    
    // Update field appearance
    if (isValid) {
        field.classList.remove('border-red-500');
        field.classList.add('border-green-500');
        removeFieldError(field);
    } else {
        field.classList.remove('border-green-500');
        field.classList.add('border-red-500');
        showFieldError(field, message);
    }
    
    return isValid;
}

// Clear field error styling
function clearFieldError(field) {
    field.classList.remove('border-red-500', 'border-green-500');
    removeFieldError(field);
}

// Show field error
function showFieldError(field, message) {
    removeFieldError(field);
    
    const errorElement = document.createElement('p');
    errorElement.className = 'error-message';
    errorElement.textContent = message;
    
    field.parentElement.parentElement.appendChild(errorElement);
}

// Remove field error
function removeFieldError(field) {
    const errorElement = field.parentElement.parentElement.querySelector('.error-message');
    if (errorElement) {
        errorElement.remove();
    }
}

// Validate entire form
function validateForm() {
    const code = document.getElementById('code').value.trim();
    const name = document.getElementById('name').value.trim();
    
    if (!code) {
        Swal.fire({
            icon: 'error',
            title: 'ข้อมูลไม่ครบถ้วน',
            text: 'กรุณาระบุรหัสคลัง',
            confirmButtonColor: '#FF6B35'
        });
        document.getElementById('code').focus();
        return false;
    }
    
    if (code.length < 2) {
        Swal.fire({
            icon: 'error',
            title: 'รหัสคลังไม่ถูกต้อง',
            text: 'รหัสคลังต้องมีอย่างน้อย 2 ตัวอักษร',
            confirmButtonColor: '#FF6B35'
        });
        document.getElementById('code').focus();
        return false;
    }
    
    if (!name) {
        Swal.fire({
            icon: 'error',
            title: 'ข้อมูลไม่ครบถ้วน',
            text: 'กรุณาระบุชื่อคลัง',
            confirmButtonColor: '#FF6B35'
        });
        document.getElementById('name').focus();
        return false;
    }
    
    if (name.length < 3) {
        Swal.fire({
            icon: 'error',
            title: 'ชื่อคลังไม่ถูกต้อง',
            text: 'ชื่อคลังต้องมีอย่างน้อย 3 ตัวอักษร',
            confirmButtonColor: '#FF6B35'
        });
        document.getElementById('name').focus();
        return false;
    }
    
    return true;
}

// Submit form with loading state
function submitForm() {
    const submitBtn = document.getElementById('submitBtn');
    const form = document.getElementById('warehouseForm');
    
    // Show loading state
    submitBtn.classList.add('loading');
    submitBtn.disabled = true;
    
    // Show confirmation
    Swal.fire({
        title: 'ยืนยันการบันทึก',
        text: 'คุณต้องการบันทึกการเปลี่ยนแปลงหรือไม่?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#FF6B35',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'บันทึก',
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return new Promise((resolve) => {
                setTimeout(() => {
                    form.submit();
                    resolve();
                }, 1000);
            });
        }
    }).then((result) => {
        if (!result.isConfirmed) {
            // Reset button state if cancelled
            submitBtn.classList.remove('loading');
            submitBtn.disabled = false;
        }
    });
}

// Update radio button states
function updateRadioStates() {
    const radioOptions = document.querySelectorAll('.radio-option');
    radioOptions.forEach(option => {
        const radio = option.querySelector('input[type="radio"]');
        if (radio && radio.checked) {
            option.classList.add('active');
        } else {
            option.classList.remove('active');
        }
    });
}

// Delete warehouse function
function deleteWarehouse() {
    Swal.fire({
        title: 'ยืนยันการลบคลัง',
        text: 'คุณต้องการลบคลังนี้หรือไม่? การดำเนินการนี้ไม่สามารถยกเลิกได้',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#EF4444',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'ลบคลัง',
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`{{ route('admin.warehouses.destroy', $warehouse) }}`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (!data.success) {
                    throw new Error(data.message || 'เกิดข้อผิดพลาด');
                }
                return data;
            })
            .catch(error => {
                Swal.showValidationMessage(`เกิดข้อผิดพลาด: ${error.message}`);
            });
        },
        allowOutsideClick: () => !Swal.isLoading()
    }).then((result) => {
        if (result.isConfirmed && result.value) {
            Swal.fire({
                icon: 'success',
                title: 'ลบคลังสำเร็จ!',
                text: result.value.message,
                confirmButtonColor: '#FF6B35'
            }).then(() => {
                window.location.href = '{{ route('admin.warehouses.index') }}';
            });
        }
    });
}

// Show success/error messages from session
@if(session('success'))
    document.addEventListener('DOMContentLoaded', function() {
        Swal.fire({
            icon: 'success',
            title: 'สำเร็จ!',
            text: '{{ session('success') }}',
            confirmButtonColor: '#FF6B35'
        });
    });
@endif

@if(session('error'))
    document.addEventListener('DOMContentLoaded', function() {
        Swal.fire({
            icon: 'error',
            title: 'เกิดข้อผิดพลาด!',
            text: '{{ session('error') }}',
            confirmButtonColor: '#FF6B35'
        });
    });
@endif

// Enhanced search functionality for users and rates
document.addEventListener('DOMContentLoaded', function() {
    // Add search functionality to user/rate lists if needed
    const userItems = document.querySelectorAll('.user-item');
    const rateItems = document.querySelectorAll('.rate-item');
    
    // Add hover effects and click interactions
    userItems.forEach(item => {
        item.addEventListener('click', function() {
            const userId = this.dataset.userId;
            if (userId) {
                window.open(`/admin/users/${userId}/edit`, '_blank');
            }
        });
    });
    
    rateItems.forEach(item => {
        item.addEventListener('click', function() {
            const rateId = this.dataset.rateId;
            if (rateId) {
                window.open(`/admin/commission-rates/${rateId}/edit`, '_blank');
            }
        });
    });
});

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    // Ctrl/Cmd + S to save
    if ((e.ctrlKey || e.metaKey) && e.key === 's') {
        e.preventDefault();
        if (validateForm()) {
            submitForm();
        }
    }
    
    // Escape to cancel
    if (e.key === 'Escape') {
        window.location.href = '{{ route('admin.warehouses.index') }}';
    }
    
    // Ctrl/Cmd + D to delete (if allowed)
    if ((e.ctrlKey || e.metaKey) && e.key === 'd' && {{ $warehouse->users->count() === 0 ? 'true' : 'false' }}) {
        e.preventDefault();
        deleteWarehouse();
    }
});

// Auto-save draft functionality
let autoSaveTimer;
let hasChanges = false;

// Track form changes
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('warehouseForm');
    const originalData = new FormData(form);
    
    // Store original values
    const originalValues = {};
    for (let [key, value] of originalData.entries()) {
        originalValues[key] = value;
    }
    
    // Track changes
    form.addEventListener('input', function() {
        checkForChanges(originalValues);
    });
    
    form.addEventListener('change', function() {
        checkForChanges(originalValues);
    });
});

function checkForChanges(originalValues) {
    const form = document.getElementById('warehouseForm');
    const currentData = new FormData(form);
    let hasChanges = false;
    
    for (let [key, value] of currentData.entries()) {
        if (originalValues[key] !== value) {
            hasChanges = true;
            break;
        }
    }
    
    // Update UI to show unsaved changes
    if (hasChanges) {
        document.title = '* แก้ไขคลังสินค้า - ระบบจัดการค่าคอมมิชชั่น';
        showUnsavedChangesIndicator();
    } else {
        document.title = 'แก้ไขคลังสินค้า - ระบบจัดการค่าคอมมิชชั่น';
        hideUnsavedChangesIndicator();
    }
}

function showUnsavedChangesIndicator() {
    let indicator = document.querySelector('.unsaved-indicator');
    if (!indicator) {
        indicator = document.createElement('div');
        indicator.className = 'unsaved-indicator';
        indicator.innerHTML = '<i class="fas fa-circle text-orange-500"></i> มีการเปลี่ยนแปลงที่ยังไม่ได้บันทึก';
        indicator.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: white;
            padding: 0.5rem 1rem;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            font-size: 0.875rem;
            z-index: 9999;
            border-left: 4px solid #FF6B35;
        `;
        document.body.appendChild(indicator);
    }
}

function hideUnsavedChangesIndicator() {
    const indicator = document.querySelector('.unsaved-indicator');
    if (indicator) {
        indicator.remove();
    }
}

// Warn before leaving if there are unsaved changes
window.addEventListener('beforeunload', function(e) {
    if (hasChanges) {
        e.preventDefault();
        e.returnValue = 'คุณมีการเปลี่ยนแปลงที่ยังไม่ได้บันทึก คุณต้องการออกจากหน้านี้หรือไม่?';
        return e.returnValue;
    }
});

// Remove warning when form is submitted
document.getElementById('warehouseForm').addEventListener('submit', function() {
    hasChanges = false;
});

// Add smooth scrolling to validation errors
function scrollToError(element) {
    element.scrollIntoView({
        behavior: 'smooth',
        block: 'center'
    });
    
    // Add pulse animation to highlight the error
    element.style.animation = 'pulse 2s ease-in-out';
    setTimeout(() => {
        element.style.animation = '';
    }, 2000);
}

// Add CSS for pulse animation
const style = document.createElement('style');
style.textContent = `
    @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.02); }
    }
`;
document.head.appendChild(style);
</script>

@endsection