@extends('layouts.app')
@push('styles')
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/aos/2.3.4/aos.css">
<style>
:root {
    --primary: #4f46e5;
    --primary-light: #6366f1;
    --success: #10b981;
    --danger: #ef4444;
    --warning: #f59e0b;
    --gray-100: #f9fafb;
    --gray-200: #f3f4f6;
    --gray-300: #e5e7eb;
    --gray-400: #9ca3af;
    --gray-500: #6b7280;
    --gray-700: #374151;
    --gray-900: #111827;
    --radius: 8px;
    --transition: all 0.2s ease-in-out;
}

/* ให้ body และ html เต็มจอ */
html, body {
    height: 100%;
    margin: 0;
}

/* ถ้า layout.app ใช้ flex หรือ grid คุม ให้คุมความสูงเต็มจอ */
body > #app {
    min-height: 100%;
    display: flex;
    flex-direction: column;
}

/* Main content */
main, .content {
    flex: 1 0 auto; /* ขยายเต็ม */
    display: flex;
    flex-direction: column;
}

/* Container ของหน้า records */
.records-container {
    flex: 1;
    width: 100%;
    min-height: 100%;
    box-sizing: border-box;
    padding: 1.5rem;
}


/* Header */
.records-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    flex-wrap: wrap;
    gap: 1rem;
    margin-bottom: 2rem;
}

.records-title {
    font-size: 1.75rem;
    color: var(--gray-900);
    margin: 0;
    display: flex;
    align-items: center;
    gap: .5rem;
}

.records-title i {
    color: var(--primary);
}

.records-subtitle {
    color: var(--gray-500);
    margin-top: .25rem;
    font-size: .95rem;
}

.header-right {
    display: flex;
    flex-wrap: wrap;
    gap: .5rem;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: .4rem;
    padding: .6rem 1rem;
    border-radius: var(--radius);
    border: none;
    font-size: .9rem;
    cursor: pointer;
    transition: var(--transition);
    font-weight: 500;
}

.btn i { font-size: .9rem; }

.btn:hover { opacity: 0.9; }

.btn-export { background: var(--success); color: #fff; }
.btn-print { background: var(--primary-light); color: #fff; }
.btn-income { background: var(--primary); color: #fff; }
.btn-expense { background: var(--danger); color: #fff; }
.btn-reset { background: var(--gray-300); color: var(--gray-700); }

.btn-sm { padding: .4rem .7rem; font-size: .8rem; }

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

/* ปรับการ์ด */
.stat-card {
    background: #fff;
    border-radius: 12px;
    box-shadow: 0 2px 6px rgba(0,0,0,0.05);
    padding: 1.2rem 1.5rem;
    display: flex;
    flex-direction: column;
    gap: 0.6rem;
    position: relative;
    border-top: 4px solid var(--primary-light);
}

/* ไอคอน */
.stat-icon {
    width: 48px;
    height: 48px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    font-size: 1.5rem;
    color: #fff;
    margin-bottom: 0.5rem;
}
.stat-card.income .stat-icon { background: var(--success); }
.stat-card.expense .stat-icon { background: var(--danger); }
.stat-card.balance .stat-icon { background: var(--primary); }
.stat-card.savings .stat-icon { background: var(--warning); }

/* ข้อความ */
.stat-label {
    font-size: 0.95rem;
    font-weight: 500;
    color: var(--gray-600);
}
.stat-value {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--gray-900);
}
.stat-change {
    font-size: 0.85rem;
    display: flex;
    align-items: center;
    gap: .3rem;
}
.stat-change.positive { color: var(--success); }
.stat-change.negative { color: var(--danger); }

/* responsive */
@media (max-width: 768px) {
    .stat-card {
        padding: 1rem;
    }
    .stat-value {
        font-size: 1.3rem;
    }
}


/* Filters */
.filters-section {
    background: #fff;
    padding: 1.2rem;
    border-radius: 12px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
    margin-bottom: 2rem;
}

/* ใช้ Grid layout */
.filters-form {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 1rem;
    align-items: end;
}

/* Label */
.filter-label {
    font-size: .85rem;
    color: var(--gray-600);
    margin-bottom: .4rem;
    display: flex;
    align-items: center;
    gap: .3rem;
}

/* Select, Input */
.form-control {
    width: 100%;
    padding: .55rem .75rem;
    border: 1px solid var(--gray-300);
    border-radius: 8px;
    font-size: .9rem;
    background: #fff;
    transition: border-color .2s ease;
}
.form-control:focus {
    border-color: var(--primary);
    outline: none;
}

/* Search Box */
.search-wrapper {
    display: flex;
    border: 1px solid var(--gray-300);
    border-radius: 8px;
    overflow: hidden;
}
.search-input {
    flex: 1;
    padding: .55rem .75rem;
    border: none;
    font-size: .9rem;
}
.search-btn {
    background: var(--primary);
    color: #fff;
    border: none;
    padding: 0 .9rem;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
}
.search-btn:hover { background: var(--primary-light); }

/* ----- Responsive สำหรับมือถือ ----- */
@media (max-width: 768px) {
    .filters-form {
        grid-template-columns: 1fr; /* 1 ช่องต่อแถว */
        gap: 1rem;
    }
    .filters-section {
        padding: 1rem;
        border-radius: 0; /* ชิดขอบจอ เหมือนในแอพ */
        box-shadow: none;
    }
}


/* Table */
/* ----- Table ปกติ (Desktop) ----- */
.records-table {
    width: 100%;
    border-collapse: collapse;
}
.records-table th, 
.records-table td {
    padding: .75rem .6rem;
    border-bottom: 1px solid var(--gray-200);
    text-align: left;
    font-size: .9rem;
}
.records-table th {
    background: var(--gray-100);
    font-weight: 600;
    color: var(--gray-700);
}

/* ----- Mobile View: เปลี่ยนเป็น Card ----- */
@media (max-width: 768px) {
    .records-table thead {
        display: none; /* ซ่อนหัวตาราง */
    }

    .records-table,
    .records-table tbody,
    .records-table tr,
    .records-table td {
        display: block;
        width: 100%;
    }

    .records-table tr {
        background: #fff;
        margin-bottom: 1rem;
        border-radius: 10px;
        box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        padding: .8rem 1rem;
    }

    .records-table td {
        border: none;
        padding: .4rem 0;
        font-size: .9rem;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    /* เพิ่ม Label ของคอลัมน์ไว้หน้าแต่ละค่า */
    .records-table td::before {
        content: attr(data-label);
        font-weight: 600;
        color: var(--gray-600);
        flex: 1;
        text-align: left;
    }

    /* ค่าต่างๆ ขยับไปขวา */
    .records-table td span,
    .records-table td div {
        flex: 1;
        text-align: right;
    }

    /* Action ปรับให้ปุ่มชิดซ้ายขวา */
    .records-table td:last-child {
        justify-content: flex-end;
        gap: .5rem;
    }
}


/* Checkbox */
.checkbox-wrapper {
    position: relative;
}
.checkbox-input {
    appearance: none;
    width: 18px; height: 18px;
    border: 1.5px solid var(--gray-300);
    border-radius: 4px;
    cursor: pointer;
    transition: var(--transition);
}
.checkbox-input:checked {
    background: var(--primary);
    border-color: var(--primary);
}
.checkbox-label {
    position: absolute;
    top: 0; left: 0;
    width: 18px; height: 18px;
    cursor: pointer;
}

/* Badges */
.type-badge {
    display: inline-flex;
    align-items: center;
    gap: .3rem;
    padding: .2rem .5rem;
    border-radius: var(--radius);
    font-size: .8rem;
    font-weight: 500;
}
.type-badge.income { background: #ecfdf5; color: var(--success); }
.type-badge.expense { background: #fef2f2; color: var(--danger); }

.source-badge {
    padding: .2rem .5rem;
    border-radius: var(--radius);
    font-size: .75rem;
}
.source-badge.manual { background: #f3f4f6; color: var(--gray-700); }
.source-badge.delivery { background: #e0f2fe; color: #0284c7; }

/* Category */
.category-display {
    display: flex;
    align-items: center;
    gap: .4rem;
    color: var(--gray-700);
}
.category-icon {
    color: var(--category-color, var(--gray-400));
}

/* Note */
.note-display {
    display: flex;
    flex-wrap: wrap;
    align-items: center;
    gap: .4rem;
}

.badge {
    font-size: .7rem;
    padding: .2rem .5rem;
    border-radius: var(--radius);
    font-weight: 500;
    display: inline-flex;
    align-items: center;
    gap: .2rem;
}
.badge-delivery { background: #e0f2fe; color: #0284c7; }
.badge-recurring { background: #fef9c3; color: #ca8a04; }

/* Amount */
.amount {
    font-weight: bold;
}
.amount.income { color: var(--success); }
.amount.expense { color: var(--danger); }

/* Actions */
.action-buttons {
    display: flex;
    justify-content: center;
    gap: .5rem;
}
.btn-action {
    background: var(--gray-100);
    padding: .4rem;
    border-radius: var(--radius);
    border: none;
    cursor: pointer;
    transition: var(--transition);
}
.btn-action:hover { background: var(--gray-200); }
.btn-action.edit { color: var(--primary); }
.btn-action.delete { color: var(--danger); }
.btn-action.duplicate { color: var(--warning); }

/* Pagination */
.pagination-wrapper {
    margin-top: 1rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: .85rem;
}

/* Empty state */
.empty-state {
    text-align: center;
    padding: 3rem 1rem;
    color: var(--gray-500);
}
.empty-icon {
    font-size: 3rem;
    color: var(--gray-300);
    margin-bottom: 1rem;
}
.empty-actions {
    margin-top: 1rem;
    display: flex;
    justify-content: center;
    gap: .5rem;
}

/* Responsive */
@media (max-width: 768px) {
    .records-header {
        flex-direction: column;
        align-items: flex-start;
    }
    .filters-form {
        flex-direction: column;
    }
    .table-actions {
        flex-direction: column;
        gap: .5rem;
    }
    .stat-value { font-size: 1.2rem; }
}



</style>
@endpush
@section('content')
<div class="records-container">
    <!-- Header -->
    <div class="records-header" data-aos="fade-down">
        <div class="header-left">
            <h1 class="records-title">
                <i class="fas fa-coins"></i>
                บันทึกรายรับ-รายจ่าย
            </h1>
            <p class="records-subtitle">จัดการรายรับรายจ่ายของคุณได้อย่างง่ายดาย</p>
        </div>
        <div class="header-right">
            <button class="btn btn-export" onclick="exportRecords()">
                <i class="fas fa-file-export"></i>
                <span class="btn-text">ส่งออก</span>
            </button>
            <button class="btn btn-print" onclick="printRecords()">
                <i class="fas fa-print"></i>
                <span class="btn-text">พิมพ์</span>
            </button>
            <a href="{{ route('finance.records.create', ['type' => 'income']) }}" class="btn btn-income">
                <i class="fas fa-plus-circle"></i>
                <span class="btn-text">เพิ่มรายรับ</span>
            </a>
            <a href="{{ route('finance.records.create', ['type' => 'expense']) }}" class="btn btn-expense">
                <i class="fas fa-minus-circle"></i>
                <span class="btn-text">เพิ่มรายจ่าย</span>
            </a>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="stats-grid" data-aos="fade-up">
    <div class="stat-card income">
        <div class="stat-icon">
            <i class="fas fa-arrow-up"></i>
        </div>
        <div class="stat-content">
            <span class="stat-label">รายรับทั้งหมด</span>
            <span class="stat-value">฿{{ number_format($totalIncome, 2) }}</span>
            <div class="stat-change {{ $incomeChange >= 0 ? 'positive' : 'negative' }}">
                <i class="fas fa-trending-{{ $incomeChange >= 0 ? 'up' : 'down' }}"></i>
                <span>{{ abs(round($incomeChange, 1)) }}% จากเดือนที่แล้ว</span>
            </div>
        </div>
    </div>

    <div class="stat-card expense">
        <div class="stat-icon">
            <i class="fas fa-arrow-down"></i>
        </div>
        <div class="stat-content">
            <span class="stat-label">รายจ่ายทั้งหมด</span>
            <span class="stat-value">฿{{ number_format($totalExpense, 2) }}</span>
            <div class="stat-change {{ $expenseChange <= 0 ? 'positive' : 'negative' }}">
                <i class="fas fa-trending-{{ $expenseChange <= 0 ? 'down' : 'up' }}"></i>
                <span>{{ abs(round($expenseChange, 1)) }}% จากเดือนที่แล้ว</span>
            </div>
        </div>
    </div>

    <div class="stat-card balance">
        <div class="stat-icon">
            <i class="fas fa-balance-scale"></i>
        </div>
        <div class="stat-content">
            <span class="stat-label">คงเหลือสุทธิ</span>
            <span class="stat-value {{ $netAmount >= 0 ? 'positive' : 'negative' }}">
                ฿{{ number_format(abs($netAmount), 2) }}
            </span>
            <div class="stat-change {{ $netAmount >= 0 ? 'positive' : 'negative' }}">
                <i class="fas fa-{{ $netAmount >= 0 ? 'check-circle' : 'exclamation-triangle' }}"></i>
                <span>{{ $netAmount >= 0 ? 'สถานะการเงินดี' : 'ควรลดรายจ่าย' }}</span>
            </div>
        </div>
    </div>

    <div class="stat-card savings">
        <div class="stat-icon">
            <i class="fas fa-piggy-bank"></i>
        </div>
        <div class="stat-content">
            <span class="stat-label">อัตราการออม</span>
            <span class="stat-value">
                {{ $totalIncome > 0 ? round(($netAmount / $totalIncome) * 100, 1) : 0 }}%
            </span>
            <div class="stat-change positive">
                <i class="fas fa-trophy"></i>
                <span>เป้าหมาย 20%</span>
            </div>
        </div>
    </div>
</div>
    <!-- End of Stats Cards -->

    <!-- Filters Section -->
    <div class="filters-section" data-aos="fade-up">
        <form method="GET" action="{{ route('finance.records.index') }}" class="filters-form" id="filtersForm">
            <div class="filter-group">
                <label class="filter-label">
                    <i class="fas fa-filter"></i> ประเภท
                </label>
                <select name="type" class="form-control" onchange="this.form.submit()">
                    <option value="">ทั้งหมด</option>
                    <option value="income" {{ request('type') == 'income' ? 'selected' : '' }}>
                        <i class="fas fa-arrow-up"></i> รายรับ
                    </option>
                    <option value="expense" {{ request('type') == 'expense' ? 'selected' : '' }}>
                        <i class="fas fa-arrow-down"></i> รายจ่าย
                    </option>
                </select>
            </div>

            <div class="filter-group">
                <label class="filter-label">
                    <i class="fas fa-tag"></i> หมวดหมู่
                </label>
                <select name="category" class="form-control" onchange="this.form.submit()">
                    <option value="">ทั้งหมด</option>
                    @if(isset($categories['income']) && count($categories['income']) > 0)
                        <optgroup label="รายรับ">
                            @foreach($categories['income'] as $cat)
                                <option value="{{ $cat->slug }}" {{ request('category') == $cat->slug ? 'selected' : '' }}>
                                    {{ $cat->name }}
                                </option>
                            @endforeach
                        </optgroup>
                    @endif
                    @if(isset($categories['expense']) && count($categories['expense']) > 0)
                        <optgroup label="รายจ่าย">
                            @foreach($categories['expense'] as $cat)
                                <option value="{{ $cat->slug }}" {{ request('category') == $cat->slug ? 'selected' : '' }}>
                                    {{ $cat->name }}
                                </option>
                            @endforeach
                        </optgroup>
                    @endif
                </select>
            </div>

            <div class="filter-group">
                <label class="filter-label">
                    <i class="fas fa-calendar"></i> เดือน
                </label>
                <input type="month" name="month" class="form-control" 
                       value="{{ request('month') }}" onchange="this.form.submit()">
            </div>

            <div class="filter-group search-group">
                <label class="filter-label">
                    <i class="fas fa-search"></i> ค้นหา
                </label>
                <div class="search-wrapper">
                    <input type="text" name="search" class="form-control search-input" 
                           placeholder="ค้นหาหมายเหตุ..." value="{{ request('search') }}">
                    <button type="submit" class="search-btn">
                        <i class="fas fa-search"></i>
                    </button>
                </div>
            </div>

            @if(request()->hasAny(['type', 'category', 'month', 'search']))
                <a href="{{ route('finance.records.index') }}" class="btn btn-reset">
                    <i class="fas fa-redo"></i> ล้างตัวกรอง
                </a>
            @endif
        </form>
    </div>

    <!-- Records Table -->
<div class="records-table-container" data-aos="fade-up">
    @if($records->count() > 0)
        <div class="table-actions">
            <button class="btn btn-sm btn-outline" onclick="toggleSelectAll()">
                <i class="fas fa-check-square"></i> เลือกทั้งหมด
            </button>
            <button class="btn btn-sm btn-danger" onclick="bulkDelete()" style="display: none;" id="bulkDeleteBtn">
                <i class="fas fa-trash"></i> <span id="selectedCount">ลบที่เลือก</span>
            </button>
        </div>

        <div class="table-responsive">
            <table class="records-table">
                <thead>
                    <tr>
                        <th width="50">
                            <div class="checkbox-wrapper">
                                <input type="checkbox" id="selectAll" class="checkbox-input">
                                <label for="selectAll" class="checkbox-label"></label>
                            </div>
                        </th>
                        <th>วันที่</th>
                        <th>ประเภท</th>
                        <th>หมวดหมู่</th>
                        <th>รายการ</th>
                        <th class="text-right">จำนวนเงิน</th>
                        <th>แหล่งที่มา</th>
                        <th class="text-center">จัดการ</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($records as $record)
                    <tr class="record-row {{ $record->type }}" data-id="{{ $record->id }}">
                        <td data-label="เลือก">
                            @if($record->source === 'manual')
                                <div class="checkbox-wrapper">
                                    <input type="checkbox" class="record-checkbox checkbox-input" 
                                           id="record-{{ $record->id }}" value="{{ $record->id }}">
                                    <label for="record-{{ $record->id }}" class="checkbox-label"></label>
                                </div>
                            @endif
                        </td>
                        <td data-label="วันที่">
                            <div class="date-display">
                                <span class="date-day">{{ $record->record_date->format('d') }}</span>
                                <span class="date-month">{{ $record->record_date->locale('th')->monthName }}</span>
                                <span class="date-year">{{ $record->record_date->format('Y') + 543 }}</span>
                            </div>
                        </td>
                        <td data-label="ประเภท">
                            <span class="type-badge {{ $record->type }}">
                                <i class="fas {{ $record->type === 'income' ? 'fa-arrow-up' : 'fa-arrow-down' }}"></i>
                                {{ $record->type === 'income' ? 'รายรับ' : 'รายจ่าย' }}
                            </span>
                        </td>
                        <td data-label="หมวดหมู่">
                            <div class="category-display" style="--category-color: {{ $record->categoryInfo->color ?? '#6B7280' }}">
                                <span class="category-icon">
                                    <i class="fas {{ $record->categoryInfo->icon ?? 'fa-circle' }}"></i>
                                </span>
                                <span class="category-name">{{ $record->categoryInfo->name ?? $record->category }}</span>
                            </div>
                        </td>
                        <td data-label="รายการ">
                            <div class="note-display">
                                <span class="note-text">{{ $record->note ?? '-' }}</span>
                                @if($record->source === 'delivery')
                                    <span class="badge badge-delivery">
                                        <i class="fas fa-truck"></i> จากการจัดส่ง
                                    </span>
                                @endif
                                @if($record->is_recurring)
                                    <span class="badge badge-recurring">
                                        <i class="fas fa-sync"></i> รายการซ้ำ
                                    </span>
                                @endif
                            </div>
                        </td>
                        <td class="text-right" data-label="จำนวนเงิน">
                            <span class="amount {{ $record->type }}">
                                <span class="amount-sign">{{ $record->type === 'income' ? '+' : '-' }}</span>
                                <span class="amount-value">฿{{ number_format($record->amount, 2) }}</span>
                            </span>
                        </td>
                        <td data-label="แหล่งที่มา">
                            <span class="source-badge {{ $record->source }}">
                                <i class="fas {{ $record->source === 'delivery' ? 'fa-robot' : 'fa-user' }}"></i>
                                {{ $record->source === 'delivery' ? 'ระบบ' : 'บันทึกเอง' }}
                            </span>
                        </td>
                        <td data-label="จัดการ">
                            <div class="action-buttons">
                                @if($record->source === 'manual')
                                    <a href="{{ route('finance.records.edit', $record) }}" 
                                       class="btn-action edit" title="แก้ไข">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    <button onclick="deleteRecord({{ $record->id }})" 
                                            class="btn-action delete" title="ลบ">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                @else
                                    <button onclick="duplicateRecord({{ $record->id }})" 
                                            class="btn-action duplicate" title="คัดลอก">
                                        <i class="fas fa-copy"></i>
                                    </button>
                                @endif
                            </div>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        @if ($records->hasPages())
            <div class="pagination-wrapper">
                <div class="pagination-info">
                    แสดง {{ $records->firstItem() }} - {{ $records->lastItem() }} 
                    จากทั้งหมด {{ number_format($records->total()) }} รายการ
                </div>
                {{ $records->links('vendor.pagination.custom') }}
            </div>
        @endif
    @else
        <!-- Empty State -->
        <div class="empty-state">
            <div class="empty-icon">
                <i class="fas fa-inbox"></i>
            </div>
            <h3 class="empty-title">ไม่พบรายการ</h3>
            <p class="empty-description">ยังไม่มีรายการตามเงื่อนไขที่เลือก</p>
            <div class="empty-actions">
                <a href="{{ route('finance.records.create', ['type' => 'income']) }}" class="btn btn-income">
                    <i class="fas fa-plus-circle"></i> เพิ่มรายรับ
                </a>
                <a href="{{ route('finance.records.create', ['type' => 'expense']) }}" class="btn btn-expense">
                    <i class="fas fa-minus-circle"></i> เพิ่มรายจ่าย
                </a>
            </div>
        </div>
    @endif
</div>
    <!-- End of Records Table -->
    </div>
</div>
@endsection

@push('styles')
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/aos/2.3.4/aos.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function () {
    // Initialize AOS
    AOS.init({
        duration: 600,
        once: true,
        offset: 50
    });

    const csrfToken = '{{ csrf_token() }}';

    // Select All functionality
    const selectAll = document.getElementById('selectAll');
    if (selectAll) {
        selectAll.addEventListener('change', function () {
            const checkboxes = document.querySelectorAll('.record-checkbox');
            checkboxes.forEach(cb => cb.checked = this.checked);
            updateBulkActions();
        });
    }

    // Individual checkbox
    document.querySelectorAll('.record-checkbox').forEach(cb => {
        cb.addEventListener('change', updateBulkActions);
    });

    function updateBulkActions() {
        const checkedBoxes = document.querySelectorAll('.record-checkbox:checked');
        const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
        if (bulkDeleteBtn) {
            if (checkedBoxes.length > 0) {
                bulkDeleteBtn.style.display = 'inline-flex';
                document.getElementById('selectedCount').textContent = `ลบที่เลือก (${checkedBoxes.length})`;
            } else {
                bulkDeleteBtn.style.display = 'none';
            }
        }
    }

    // Toggle select all
    window.toggleSelectAll = function () {
        if (selectAll) {
            selectAll.checked = !selectAll.checked;
            selectAll.dispatchEvent(new Event('change'));
        }
    }

    // Delete single record
    window.deleteRecord = function (id) {
        Swal.fire({
            title: 'ยืนยันการลบ?',
            text: 'คุณต้องการลบรายการนี้ใช่หรือไม่?',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#ef4444',
            cancelButtonColor: '#6b7280',
            confirmButtonText: 'ใช่, ลบเลย!',
            cancelButtonText: 'ยกเลิก'
        }).then((result) => {
            if (result.isConfirmed) {
                fetch(`{{ route('finance.records.destroy', ':id') }}`.replace(':id', id), {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': csrfToken,
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'สำเร็จ!',
                            text: 'ลบรายการเรียบร้อยแล้ว',
                            timer: 1500,
                            showConfirmButton: false
                        }).then(() => location.reload());
                    } else {
                        Swal.fire('เกิดข้อผิดพลาด!', data.message || 'ไม่สามารถลบรายการได้', 'error');
                    }
                })
                .catch(() => Swal.fire('เกิดข้อผิดพลาด!', 'ไม่สามารถติดต่อเซิร์ฟเวอร์ได้', 'error'));
            }
        });
    }

    // Bulk delete
    window.bulkDelete = function () {
        const checkedBoxes = document.querySelectorAll('.record-checkbox:checked');
        const ids = Array.from(checkedBoxes).map(cb => cb.value);
        if (ids.length === 0) return;

        Swal.fire({
            title: 'ยืนยันการลบ?',
            text: `คุณต้องการลบ ${ids.length} รายการที่เลือกใช่หรือไม่?`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#ef4444',
            cancelButtonColor: '#6b7280',
            confirmButtonText: 'ใช่, ลบทั้งหมด!',
            cancelButtonText: 'ยกเลิก'
        }).then((result) => {
            if (result.isConfirmed) {
                fetch('{{ route('finance.records.bulk-delete') }}', {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': csrfToken,
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ ids })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'สำเร็จ!',
                            text: data.message,
                            timer: 1500,
                            showConfirmButton: false
                        }).then(() => location.reload());
                    } else {
                        Swal.fire('เกิดข้อผิดพลาด!', data.message || 'ไม่สามารถลบรายการได้', 'error');
                    }
                })
                .catch(() => Swal.fire('เกิดข้อผิดพลาด!', 'ไม่สามารถติดต่อเซิร์ฟเวอร์ได้', 'error'));
            }
        });
    }

    // Duplicate record
    window.duplicateRecord = function (id) {
        Swal.fire({
            title: 'คัดลอกรายการ',
            text: 'คุณต้องการคัดลอกรายการนี้ใช่หรือไม่?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#667eea',
            cancelButtonColor: '#6b7280',
            confirmButtonText: 'ใช่, คัดลอก!',
            cancelButtonText: 'ยกเลิก'
        }).then((result) => {
            if (result.isConfirmed) {
                fetch(`{{ route('finance.records.duplicate', ':id') }}`.replace(':id', id), {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': csrfToken,
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'สำเร็จ!',
                            text: 'คัดลอกรายการเรียบร้อยแล้ว',
                            timer: 1500,
                            showConfirmButton: false
                        }).then(() => {
                            if (data.redirect) {
                                window.location.href = data.redirect;
                            } else {
                                location.reload();
                            }
                        });
                    } else {
                        Swal.fire('เกิดข้อผิดพลาด!', data.message || 'ไม่สามารถคัดลอกรายการได้', 'error');
                    }
                })
                .catch(() => Swal.fire('เกิดข้อผิดพลาด!', 'ไม่สามารถติดต่อเซิร์ฟเวอร์ได้', 'error'));
            }
        });
    }

    // Export records
    window.exportRecords = function () {
        Swal.fire({
            title: 'เลือกรูปแบบการส่งออก',
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'CSV',
            cancelButtonText: 'PDF',
            confirmButtonColor: '#10b981',
            cancelButtonColor: '#667eea',
        }).then((result) => {
            if (result.isConfirmed) {
                window.location.href = '{{ route('finance.reports.export') }}?type=monthly&format=csv&period={{ now()->format('Y-m') }}';
            } else if (result.dismiss === Swal.DismissReason.cancel) {
                window.location.href = '{{ route('finance.reports.export') }}?type=monthly&format=pdf&period={{ now()->format('Y-m') }}';
            }
        });
    }

    // Print records
    window.printRecords = function () {
        window.print();
    }

    // Keyboard shortcuts
    document.addEventListener('keydown', function (e) {
        if ((e.ctrlKey || e.metaKey) && e.key === 'n') {
            e.preventDefault();
            window.location.href = '{{ route('finance.records.create') }}';
        }
        if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
            e.preventDefault();
            const searchInput = document.querySelector('.search-input');
            if (searchInput) searchInput.focus();
        }
    });

    // Auto-submit form on input change with debounce
    const searchInput = document.querySelector('.search-input');
    if (searchInput) {
        let searchTimer;
        searchInput.addEventListener('input', function () {
            clearTimeout(searchTimer);
            searchTimer = setTimeout(() => this.form.submit(), 500);
        });
    }

    // Smooth scroll
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) target.scrollIntoView({ behavior: 'smooth', block: 'start' });
        });
    });

    // Tooltips
    document.querySelectorAll('[title]').forEach(element => {
        element.addEventListener('mouseenter', function () {
            this.setAttribute('data-title', this.getAttribute('title'));
            this.removeAttribute('title');
        });
        element.addEventListener('mouseleave', function () {
            this.setAttribute('title', this.getAttribute('data-title'));
            this.removeAttribute('data-title');
        });
    });
});
</script>


@endpush