<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Cache;

class TranslationController extends Controller
{
    /**
     * Get translations for a specific locale
     */
    public function getTranslations($locale)
    {
        if (!in_array($locale, config('app.available_locales', ['th', 'en']))) {
            return response()->json(['error' => 'Invalid locale'], 400);
        }
        
        $cacheKey = "translations.{$locale}";
        
        return Cache::remember($cacheKey, 3600, function () use ($locale) {
            $translations = [];
            // Validate and sanitize locale path
            $safeLangPath = resource_path('lang/' . basename($locale));
            $langPath = realpath($safeLangPath);
            
            // Ensure path is within allowed directory
            if (!$langPath || !str_starts_with($langPath, realpath(resource_path('lang')))) {
                return [];
            }
            
            if (File::exists($langPath)) {
                $files = File::files($langPath);
                
                foreach ($files as $file) {
                    $filename = $file->getFilenameWithoutExtension();
                    $content = include $file->getPathname();
                    
                    if (is_array($content)) {
                        $translations[$filename] = $content;
                    }
                }
            }
            
            // Also load JSON translations
            $jsonFile = resource_path("lang/{$locale}.json");
            if (File::exists($jsonFile)) {
                $jsonContent = json_decode(File::get($jsonFile), true);
                if (is_array($jsonContent)) {
                    $translations = array_merge($translations, $jsonContent);
                }
            }
            
            return $translations;
        });
    }
    
    /**
     * Clear translation cache
     */
    public function clearCache()
    {
        $locales = config('app.available_locales', ['th', 'en']);
        
        foreach ($locales as $locale) {
            Cache::forget("translations.{$locale}");
        }
        
        // Clear used keys cache
        Cache::forget('translation_used_keys');
        
        return response()->json(['success' => true, 'message' => 'Translation cache cleared']);
    }
    
    /**
     * Get missing translation keys
     */
    public function getMissingKeys()
    {
        $locales = config('app.available_locales', ['th', 'en']);
        $missing = [];
        
        foreach ($locales as $locale) {
            $translations = $this->getTranslations($locale)->getData();
            $missing[$locale] = $this->findMissingKeys($translations);
        }
        
        return response()->json($missing);
    }
    
    /**
     * Find missing translation keys by scanning files
     */
    private function findMissingKeys($translations)
    {
        $cacheKey = 'translation_used_keys';
        
        $usedKeys = Cache::remember($cacheKey, 1800, function () {
            $keys = [];
            
            // Scan Blade files
            $bladeFiles = File::allFiles(resource_path('views'));
            foreach ($bladeFiles as $file) {
                if ($file->getExtension() === 'php') {
                    $content = File::get($file);
                    preg_match_all('/__\([\'"]([^\'"]+)[\'"]\)/', $content, $matches);
                    $keys = array_merge($keys, $matches[1]);
                }
            }
            
            // Scan PHP files
            $phpFiles = File::allFiles(app_path());
            foreach ($phpFiles as $file) {
                if ($file->getExtension() === 'php') {
                    $content = File::get($file);
                    preg_match_all('/__\([\'"]([^\'"]+)[\'"]\)/', $content, $matches);
                    $keys = array_merge($keys, $matches[1]);
                }
            }
            
            return array_unique($keys);
        });
        
        $missing = [];
        foreach ($usedKeys as $key) {
            if (!$this->hasTranslation($key, $translations)) {
                $missing[] = $key;
            }
        }
        
        return $missing;
    }
    
    /**
     * Check if translation exists
     */
    private function hasTranslation($key, $translations)
    {
        $keys = explode('.', $key);
        $value = $translations;
        
        foreach ($keys as $k) {
            if (is_array($value) && array_key_exists($k, $value)) {
                $value = $value[$k];
            } else {
                return false;
            }
        }
        
        return true;
    }
}