<?php

namespace App\Http\Controllers;

use App\Models\DeliveryRecord;
use App\Models\CommissionRate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class VehicleDashboardController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        // Ensure user has vehicle role
        if (!in_array($user->role, ['2wheel', '4wheel'])) {
            abort(403, 'คุณไม่มีสิทธิ์เข้าถึงหน้านี้');
        }

        // Get today's stats
        $todayStats = $this->getTodayStats($user->id);
        
        // Get this week's stats
        $weekStats = $this->getWeekStats($user->id);
        
        // Get this month's stats
        $monthStats = $this->getMonthStats($user->id);
        
        // Get average stats
        $avgStats = $this->getAverageStats($user->id, $monthStats);
        
        // Get recent delivery records
        $recentRecords = DeliveryRecord::where('user_id', $user->id)
            ->orderBy('delivery_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
        
        // Get commission rates for user's warehouse and vehicle type
        $warehouseCode = $user->warehouse_code ?? '';
        $vehicleType = $user->role ?? '';
        
        $commissionRates = CommissionRate::where('warehouse_code', '=', $warehouseCode)
            ->where('vehicle_type', '=', $vehicleType)
            ->where('is_active', true)
            ->orderBy('tier')
            ->get();
        
        // Get chart data for last 7 days
        $chartData = $this->getChartData($user->id);
        
        return view('vehicle.dashboard', compact(
            'todayStats',
            'weekStats',
            'monthStats',
            'avgStats',
            'recentRecords',
            'commissionRates',
            'chartData'
        ));
    }
    
    private function getTodayStats($userId)
    {
        $today = Carbon::today();
        
        $todayRecord = DeliveryRecord::where('user_id', $userId)
            ->whereDate('delivery_date', $today)
            ->first();
        
        return [
            'parcels' => $todayRecord->parcel_count ?? 0,
            'commission' => $todayRecord->commission_amount ?? 0,
        ];
    }
    
    private function getWeekStats($userId)
    {
        $startOfWeek = Carbon::now()->startOfWeek();
        $endOfWeek = Carbon::now()->endOfWeek();
        
        $weekStats = DeliveryRecord::where('user_id', $userId)
            ->whereBetween('delivery_date', [$startOfWeek, $endOfWeek])
            ->selectRaw('SUM(parcel_count) as parcels, SUM(commission_amount) as commission, COUNT(*) as days')
            ->first();
        
        return [
            'parcels' => $weekStats->parcels ?? 0,
            'commission' => $weekStats->commission ?? 0,
            'days' => $weekStats->days ?? 0,
        ];
    }
    
    private function getMonthStats($userId)
    {
        $startOfMonth = Carbon::now()->startOfMonth();
        $endOfMonth = Carbon::now()->endOfMonth();
        
        $monthStats = DeliveryRecord::where('user_id', $userId)
            ->whereBetween('delivery_date', [$startOfMonth, $endOfMonth])
            ->selectRaw('SUM(parcel_count) as parcels, SUM(commission_amount) as commission, COUNT(*) as days')
            ->first();
        
        return [
            'parcels' => $monthStats->parcels ?? 0,
            'commission' => $monthStats->commission ?? 0,
            'days' => $monthStats->days ?? 0,
        ];
    }
    
    private function getAverageStats($userId, $monthStats)
    {
        $days = $monthStats['days'] > 0 ? $monthStats['days'] : 1;
        
        return [
            'parcels_per_day' => $monthStats['parcels'] / $days,
            'commission_per_day' => $monthStats['commission'] / $days,
        ];
    }
    
    private function getChartData($userId)
    {
        $labels = [];
        $parcels = [];
        $commissions = [];
        
        // Get data for last 7 days
        for ($i = 6; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            
            $record = DeliveryRecord::where('user_id', $userId)
                ->whereDate('delivery_date', $date)
                ->first();
            
            $labels[] = $date->format('d/m');
            $parcels[] = $record->parcel_count ?? 0;
            $commissions[] = $record->commission_amount ?? 0;
        }
        
        return [
            'labels' => $labels,
            'parcels' => $parcels,
            'commissions' => $commissions,
        ];
    }
    
    /**
     * Calculate commission for the calculator page
     */
    public function calculateCommission(Request $request)
    {
        $user = Auth::user();
        
        if ($request->isMethod('post')) {
            $request->validate([
                'parcel_count' => 'required|integer|min:1',
            ]);
            
            $result = CommissionRate::calculateCommission(
                $user->warehouse_code ?? 'DEFAULT',
                $user->role,
                $request->parcel_count
            );
            
            return view('vehicle.calculator', compact('result'));
        }
        
        // Get commission rates for display
        $commissionRates = CommissionRate::where('warehouse_code', $user->warehouse_code ?? 'DEFAULT')
            ->where('vehicle_type', $user->role)
            ->where('is_active', true)
            ->orderBy('tier')
            ->get();
        
        return view('vehicle.calculator', compact('commissionRates'));
    }
    
    /**
     * API endpoint for weekly comparison data
     */
    public function getWeeklyComparison(Request $request)
    {
        $user = Auth::user();
        
        // Current week
        $currentWeekStart = Carbon::now()->startOfWeek();
        $currentWeekEnd = Carbon::now()->endOfWeek();
        
        $currentWeekData = DeliveryRecord::where('user_id', $user->id)
            ->whereBetween('delivery_date', [$currentWeekStart, $currentWeekEnd])
            ->get();
        
        // Previous week
        $previousWeekStart = Carbon::now()->subWeek()->startOfWeek();
        $previousWeekEnd = Carbon::now()->subWeek()->endOfWeek();
        
        $previousWeekData = DeliveryRecord::where('user_id', $user->id)
            ->whereBetween('delivery_date', [$previousWeekStart, $previousWeekEnd])
            ->get();
        
        return response()->json([
            'current' => [
                'parcels' => $currentWeekData->sum('parcel_count'),
                'commission' => $currentWeekData->sum('commission_amount'),
            ],
            'previous' => [
                'parcels' => $previousWeekData->sum('parcel_count'),
                'commission' => $previousWeekData->sum('commission_amount'),
            ],
        ]);
    }
    
    /**
     * API endpoint for monthly statistics
     */
    public function getMonthlyStats(Request $request)
    {
        $user = Auth::user();
        $month = $request->input('month', now()->format('Y-m'));
        
        $startDate = Carbon::parse($month)->startOfMonth();
        $endDate = Carbon::parse($month)->endOfMonth();
        
        $records = DeliveryRecord::where('user_id', $user->id)
            ->whereBetween('delivery_date', [$startDate, $endDate])
            ->get();
        
        // Group by week
        $weeklyData = [];
        foreach ($records as $record) {
            $week = $record->delivery_date->weekOfMonth;
            if (!isset($weeklyData[$week])) {
                $weeklyData[$week] = [
                    'parcels' => 0,
                    'commission' => 0,
                    'days' => 0,
                ];
            }
            $weeklyData[$week]['parcels'] += $record->parcel_count;
            $weeklyData[$week]['commission'] += $record->commission_amount;
            $weeklyData[$week]['days']++;
        }
        
        return response()->json([
            'total' => [
                'parcels' => $records->sum('parcel_count'),
                'commission' => $records->sum('commission_amount'),
                'days' => $records->count(),
            ],
            'weekly' => $weeklyData,
        ]);
    }
    
    /**
     * API endpoint for daily chart data
     */
    public function getDailyChart(Request $request)
    {
        $user = Auth::user();
        $days = $request->input('days', 7);
        
        $data = [];
        
        for ($i = $days - 1; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            
            $record = DeliveryRecord::where('user_id', $user->id)
                ->whereDate('delivery_date', $date)
                ->first();
            
            $data[] = [
                'date' => $date->format('Y-m-d'),
                'label' => $date->format('d/m'),
                'day' => $date->locale('th')->dayName,
                'parcels' => $record->parcel_count ?? 0,
                'commission' => $record->commission_amount ?? 0,
            ];
        }
        
        return response()->json($data);
    }
}