<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class DeliveryRecord extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'warehouse_id',
        'delivery_date',
        'parcel_count',
        'commission_amount',
    ];

    protected $casts = [
        'delivery_date' => 'date',
        'parcel_count' => 'integer',
        'commission_amount' => 'decimal:2',
    ];

    /**
     * Get the user that owns the delivery record
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the warehouse associated with the delivery record
     */
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    /**
     * Scope a query to only include records for a specific month
     */
    public function scopeForMonth($query, $month)
    {
        return $query->whereRaw("DATE_FORMAT(delivery_date, '%Y-%m') = ?", [$month]);
    }

    /**
     * Scope a query to only include records for a specific user
     */
    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope a query to only include records within a date range
     */
    public function scopeBetweenDates($query, $startDate, $endDate)
    {
        return $query->whereBetween('delivery_date', [$startDate, $endDate]);
    }

    /**
     * Get the delivery date in Thai format
     */
    public function getFormattedDeliveryDateAttribute()
    {
        return Carbon::parse($this->delivery_date)->format('d/m/Y');
    }

    /**
     * Get the day name in Thai
     */
    public function getDeliveryDayNameAttribute()
    {
        $days = ['อาทิตย์', 'จันทร์', 'อังคาร', 'พุธ', 'พฤหัสบดี', 'ศุกร์', 'เสาร์'];
        return $days[Carbon::parse($this->delivery_date)->dayOfWeek];
    }

    /**
     * Get commission rate per parcel
     */
    public function getCommissionRateAttribute()
    {
        return $this->parcel_count > 0 ? $this->commission_amount / $this->parcel_count : 0;
    }

    /**
     * Check if this record is for today
     */
    public function getIstodayAttribute()
    {
        return Carbon::parse($this->delivery_date)->isToday();
    }

    /**
     * Check if this record is for this week
     */
    public function getIsThisWeekAttribute()
    {
        return Carbon::parse($this->delivery_date)->isCurrentWeek();
    }

    /**
     * Check if this record is for this month
     */
    public function getIsThisMonthAttribute()
    {
        return Carbon::parse($this->delivery_date)->isCurrentMonth();
    }
}