<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class FinanceCategory extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'type',
        'icon',
        'color',
        'sort_order',
        'is_active'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeIncome($query)
    {
        return $query->where('type', 'income');
    }

    public function scopeExpense($query)
    {
        return $query->where('type', 'expense');
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('name');
    }

    /**
     * Relationships
     */
    public function records()
    {
        return $this->hasMany(FinanceRecord::class, 'category', 'slug');
    }

    /**
     * Static methods
     */
    public static function getGroupedCategories()
    {
        $categories = self::active()->ordered()->get();

        return [
            'income' => $categories->where('type', 'income'),
            'expense' => $categories->where('type', 'expense'),
        ];
    }

    public static function getCategoryOptions($type = null)
    {
        $query = self::active()->ordered();

        if ($type) {
            $query->where('type', $type);
        }

        return $query->pluck('name', 'slug');
    }

    /**
     * Attributes
     */
    public function getIconClassAttribute()
    {
        return $this->icon ?? 'fa-circle';
    }

    public function getStyleAttribute()
    {
        return "color: {$this->color};";
    }

    public function getBadgeStyleAttribute()
    {
        return "background-color: {$this->color}20; color: {$this->color};";
    }
}