<?php
// app/Models/User.php - เพิ่ม relationships ที่ขาดหาย
namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    use HasFactory, Notifiable;

    protected $fillable = [
        'warehouse_code',
        'is_active',
        'first_name',
        'last_name',
        'email',
        'password',
        'role',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'is_active' => 'boolean',
        'password' => 'hashed',
    ];

    // Relationship กับ Warehouse
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class, 'warehouse_code', 'code');
    }

    // Relationship กับ DeliveryRecords
    public function deliveryRecords()
    {
        return $this->hasMany(DeliveryRecord::class);
    }

    // Accessor สำหรับชื่อเต็ม
    public function getNameAttribute()
    {
        return trim($this->first_name . ' ' . $this->last_name);
    }

    // Accessor สำหรับชื่อคลัง
    public function getWarehouseNameAttribute()
    {
        return $this->warehouse ? $this->warehouse->name : $this->warehouse_code;
    }

    // Helper methods สำหรับ role
    public function isAdmin()
    {
        return $this->role === 'admin';
    }

    public function is2Wheel()
    {
        return $this->role === '2wheel';
    }

    public function is4Wheel()
    {
        return $this->role === '4wheel';
    }

    public function isVehicleUser()
    {
        return in_array($this->role, ['2wheel', '4wheel']);
    }

    // Helper method สำหรับสถานะ active
    public function isActive()
    {
        return $this->is_active === true;
    }

    // Helper method สำหรับแสดงชื่อ role ภาษาไทย
    public function getRoleDisplayNameAttribute()
    {
        return match($this->role) {
            'admin' => 'ผู้ดูแลระบบ',
            '2wheel' => 'รถ 2 ล้อ',
            '4wheel' => 'รถ 4 ล้อ',
            default => ucfirst($this->role)
        };
    }

    // Scope สำหรับผู้ใช้ที่ใช้งานอยู่
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // Scope สำหรับผู้ใช้ตาม role
    public function scopeByRole($query, $role)
    {
        return $query->where('role', $role);
    }
}