<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Blade;
use App\Helpers\LocaleHelper;
use Carbon\Carbon;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        // Register LocaleHelper as singleton
        $this->app->singleton('locale', function () {
            return new LocaleHelper();
        });
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Register observers
        \App\Models\User::observe(\App\Observers\UserObserver::class);
        
        // Set locale from session
        $locale = Session::get('locale', config('app.locale'));
        if (in_array($locale, config('app.available_locales', ['th', 'en']))) {
            App::setLocale($locale);
            Carbon::setLocale($locale);
        }
        
        // Register Blade directives for i18n
        Blade::directive('currency', function ($expression) {
            return "<?php echo App\\Helpers\\LocaleHelper::formatCurrency($expression); ?>";
        });
        
        Blade::directive('number', function ($expression) {
            return "<?php echo App\\Helpers\\LocaleHelper::formatNumber($expression); ?>";
        });
        
        Blade::directive('localeDate', function ($expression) {
            return "<?php echo App\\Helpers\\LocaleHelper::formatDate($expression); ?>";
        });
        
        Blade::directive('localeDateTime', function ($expression) {
            return "<?php echo App\\Helpers\\LocaleHelper::formatDateTime($expression); ?>";
        });
        
        // Share locale data with all views
        view()->composer('*', function ($view) {
            $view->with([
                'currentLocale' => App::getLocale(),
                'availableLocales' => config('app.available_locales', ['th', 'en']),
                'localeNames' => config('app.locale_names', []),
                'isRtl' => LocaleHelper::getDirection() === 'rtl'
            ]);
        });
    }
}
