<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // ตารางหลักสำหรับบันทึกรายรับ-รายจ่าย
        Schema::create('finance_records', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->date('record_date');
            $table->enum('type', ['income', 'expense']);
            $table->string('category', 50);
            $table->decimal('amount', 12, 2);
            $table->text('note')->nullable();
            $table->enum('source', ['manual', 'delivery'])->default('manual');
            $table->unsignedBigInteger('ref_id')->nullable(); // Link to delivery_records.id
            $table->boolean('is_recurring')->default(false);
            $table->enum('recurring_type', ['daily', 'weekly', 'monthly', 'yearly'])->nullable();
            $table->timestamps();

            // Indexes
            $table->index(['user_id', 'record_date']);
            $table->index(['user_id', 'type']);
            $table->index(['user_id', 'category']);
            $table->index(['source', 'ref_id']);
        });

        // ตารางหมวดหมู่
        Schema::create('finance_categories', function (Blueprint $table) {
            $table->id();
            $table->string('name', 50);
            $table->string('slug', 50)->unique();
            $table->enum('type', ['income', 'expense']);
            $table->string('icon', 50)->nullable();
            $table->string('color', 7)->default('#6B7280');
            $table->integer('sort_order')->default(0);
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        // ตารางงบประมาณรายเดือน
        Schema::create('monthly_budgets', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->string('month', 7); // Format: YYYY-MM
            $table->string('category', 50);
            $table->decimal('budget_amount', 12, 2);
            $table->decimal('spent_amount', 12, 2)->default(0);
            $table->timestamps();

            $table->unique(['user_id', 'month', 'category']);
            $table->index(['user_id', 'month']);
        });

        // ตารางเป้าหมายการออม
        Schema::create('saving_goals', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->string('goal_name');
            $table->decimal('target_amount', 12, 2);
            $table->decimal('current_amount', 12, 2)->default(0);
            $table->date('target_date')->nullable();
            $table->string('status', 20)->default('active'); // active, completed, cancelled
            $table->timestamps();
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('saving_goals');
        Schema::dropIfExists('monthly_budgets');
        Schema::dropIfExists('finance_categories');
        Schema::dropIfExists('finance_records');
    }
};