@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    initializeUserManagement();
});

function initializeUserManagement() {
    initializeCheckboxes();
    initializeTableInteractions();
    initializeResponsive();
    initializeFilters();
    setupEventListeners();
}

// Checkbox Management
function initializeCheckboxes() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('.user-checkbox');
    
    if (selectAll) {
        selectAll.addEventListener('change', function() {
            checkboxes.forEach(checkbox => {
                checkbox.checked = this.checked;
            });
            updateBulkActions();
        });
    }
    
    checkboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            updateBulkActions();
            updateSelectAllState();
        });
    });
}

function updateSelectAllState() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('.user-checkbox');
    const checkedBoxes = document.querySelectorAll('.user-checkbox:checked');
    
    if (selectAll) {
        if (checkedBoxes.length === 0) {
            selectAll.checked = false;
            selectAll.indeterminate = false;
        } else if (checkedBoxes.length === checkboxes.length) {
            selectAll.checked = true;
            selectAll.indeterminate = false;
        } else {
            selectAll.checked = false;
            selectAll.indeterminate = true;
        }
    }
}

// Bulk Actions
function updateBulkActions() {
    const selectedCheckboxes = document.querySelectorAll('.user-checkbox:checked');
    const bulkActions = document.getElementById('bulkActions');
    const selectedCount = document.getElementById('selectedCount');
    
    if (selectedCheckboxes.length > 0) {
        bulkActions.style.display = 'flex';
        selectedCount.textContent = selectedCheckboxes.length;
        
        if (!bulkActions.classList.contains('show')) {
            bulkActions.style.opacity = '0';
            bulkActions.style.transform = 'translateY(-10px)';
            bulkActions.classList.add('show');
            
            setTimeout(() => {
                bulkActions.style.transition = 'all 0.3s ease';
                bulkActions.style.opacity = '1';
                bulkActions.style.transform = 'translateY(0)';
            }, 50);
        }
    } else {
        bulkActions.style.display = 'none';
        bulkActions.classList.remove('show');
    }
}

function clearSelection() {
    document.querySelectorAll('.user-checkbox').forEach(checkbox => {
        checkbox.checked = false;
    });
    const selectAll = document.getElementById('selectAll');
    if (selectAll) {
        selectAll.checked = false;
        selectAll.indeterminate = false;
    }
    updateBulkActions();
    showToast('info', 'ยกเลิกการเลือกแล้ว');
}

// Filter Functionality
function toggleFilters() {
    const filterContent = document.getElementById('filterContent');
    const filterToggle = document.querySelector('.filter-toggle i');
    
    if (filterContent.classList.contains('show')) {
        filterContent.classList.remove('show');
        filterToggle.style.transform = 'rotate(0deg)';
    } else {
        filterContent.classList.add('show');
        filterToggle.style.transform = 'rotate(180deg)';
    }
}

function clearFilters() {
    window.location.href = '{{ route("admin.users.index") }}';
}

function filterByStatus(status) {
    const url = new URL(window.location);
    url.searchParams.set('status', status);
    window.location.href = url.toString();
}

function filterByRole(role) {
    const url = new URL(window.location);
    url.searchParams.set('role', role);
    window.location.href = url.toString();
}

// Mobile Menu
function toggleCardMenu(userId) {
    const menu = document.getElementById('card-actions-' + userId);
    const allMenus = document.querySelectorAll('.card-actions');
    
    allMenus.forEach(m => {
        if (m.id !== 'card-actions-' + userId) {
            m.style.display = 'none';
        }
    });
    
    if (menu.style.display === 'none' || !menu.style.display) {
        menu.style.display = 'flex';
        menu.style.animation = 'slideDown 0.3s ease-out';
    } else {
        menu.style.display = 'none';
    }
}

// Toast Notifications
function showToast(icon, title, timer = 3000) {
    const Toast = Swal.mixin({
        toast: true,
        position: 'top-end',
        showConfirmButton: false,
        timer: timer,
        timerProgressBar: true,
        didOpen: (toast) => {
            toast.addEventListener('mouseenter', Swal.stopTimer);
            toast.addEventListener('mouseleave', Swal.resumeTimer);
        }
    });
    
    Toast.fire({
        icon: icon,
        title: title
    });
}

// User Actions
function exportUsers() {
    const params = new URLSearchParams(window.location.search);
    const exportUrl = `{{ route('admin.users.export') }}?${params.toString()}`;
    
    Swal.fire({
        title: 'กำลังเตรียมข้อมูล',
        text: 'กรุณารอสักครู่...',
        allowOutsideClick: false,
        didOpen: () => {
            Swal.showLoading();
        }
    });
    
    const link = document.createElement('a');
    link.href = exportUrl;
    link.download = `users_${new Date().toISOString().split('T')[0]}.csv`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    setTimeout(() => {
        Swal.close();
        showToast('success', 'ส่งออกข้อมูลสำเร็จ!');
    }, 1000);
}

function bulkAction(action) {
    const selectedIds = Array.from(document.querySelectorAll('.user-checkbox:checked')).map(cb => cb.value);
    
    if (selectedIds.length === 0) {
        Swal.fire({
            icon: 'warning',
            title: 'ยังไม่ได้เลือกผู้ใช้',
            text: 'กรุณาเลือกผู้ใช้อย่างน้อย 1 คน',
            confirmButtonColor: '#FF6B35'
        });
        return;
    }

    const actionConfig = {
        activate: {
            title: 'เปิดใช้งานผู้ใช้',
            text: `คุณต้องการเปิดใช้งานผู้ใช้ ${selectedIds.length} คนที่เลือกหรือไม่?`,
            icon: 'question',
            confirmButtonText: 'เปิดใช้งาน',
            color: '#22C55E'
        },
        deactivate: {
            title: 'ปิดใช้งานผู้ใช้',
            text: `คุณต้องการปิดใช้งานผู้ใช้ ${selectedIds.length} คนที่เลือกหรือไม่?`,
            icon: 'warning',
            confirmButtonText: 'ปิดใช้งาน',
            color: '#F59E0B'
        }
    };
    
    const config = actionConfig[action];
    
    Swal.fire({
        title: config.title,
        text: config.text,
        icon: config.icon,
        showCancelButton: true,
        confirmButtonColor: config.color,
        cancelButtonColor: '#6B7280',
        confirmButtonText: config.confirmButtonText,
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`{{ route('admin.users.bulk') }}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    action: action,
                    user_ids: selectedIds
                })
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(data => {
                if (!data.success) {
                    throw new Error(data.message || 'เกิดข้อผิดพลาด');
                }
                return data;
            })
            .catch(error => {
                Swal.showValidationMessage(`เกิดข้อผิดพลาด: ${error.message}`);
            });
        },
        allowOutsideClick: () => !Swal.isLoading()
    }).then((result) => {
        if (result.isConfirmed && result.value) {
            Swal.fire({
                icon: 'success',
                title: 'สำเร็จ!',
                text: result.value.message,
                confirmButtonColor: '#FF6B35'
            }).then(() => {
                location.reload();
            });
        }
    });
}

function toggleUserStatus(userId) {
    Swal.fire({
        title: 'ยืนยันการเปลี่ยนสถานะ',
        text: 'คุณต้องการเปลี่ยนสถานะการใช้งานของผู้ใช้นี้หรือไม่?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#FF6B35',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'ยืนยัน',
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`{{ route('admin.users.index') }}/${userId}/toggle`, {
                method: 'PATCH',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (!data.success) {
                    throw new Error(data.message || 'เกิดข้อผิดพลาด');
                }
                return data;
            })
            .catch(error => {
                Swal.showValidationMessage(`เกิดข้อผิดพลาด: ${error.message}`);
            });
        }
    }).then((result) => {
        if (result.isConfirmed && result.value) {
            Swal.fire({
                icon: 'success',
                title: 'สำเร็จ!',
                text: result.value.message,
                confirmButtonColor: '#FF6B35'
            }).then(() => {
                location.reload();
            });
        }
    });
}

function resetPassword(userId) {
    Swal.fire({
        title: 'ยืนยันการรีเซ็ตรหัสผ่าน',
        text: 'คุณต้องการรีเซ็ตรหัสผ่านของผู้ใช้นี้หรือไม่?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#F59E0B',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'รีเซ็ตรหัสผ่าน',
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`{{ route('admin.users.index') }}/${userId}/reset-password`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (!data.success) {
                    throw new Error(data.message || 'เกิดข้อผิดพลาด');
                }
                return data;
            })
            .catch(error => {
                Swal.showValidationMessage(`เกิดข้อผิดพลาด: ${error.message}`);
            });
        }
    }).then((result) => {
        if (result.isConfirmed && result.value) {
            Swal.fire({
                icon: 'success',
                title: 'รีเซ็ตรหัสผ่านสำเร็จ!',
                html: `
                    <div class="text-left bg-gray-100 p-4 rounded-lg mt-4">
                        <div class="font-semibold mb-2">รหัสผ่านใหม่:</div>
                        <div class="font-mono text-lg bg-white p-2 rounded border-2 border-dashed border-orange-300">
                            ${result.value.new_password}
                        </div>
                        <div class="text-sm text-gray-600 mt-2">
                            กรุณาบันทึกรหัสผ่านนี้และแจ้งให้ผู้ใช้ทราบ
                        </div>
                    </div>
                `,
                confirmButtonColor: '#FF6B35',
                confirmButtonText: 'เข้าใจแล้ว'
            });
        }
    });
}

function deleteUser(userId) {
    Swal.fire({
        title: 'ยืนยันการลบผู้ใช้',
        text: 'คุณต้องการลบผู้ใช้นี้หรือไม่? การดำเนินการนี้ไม่สามารถยกเลิกได้',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#EF4444',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'ลบผู้ใช้',
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`{{ route('admin.users.index') }}/${userId}`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (!data.success) {
                    throw new Error(data.message || 'เกิดข้อผิดพลาด');
                }
                return data;
            })
            .catch(error => {
                Swal.showValidationMessage(`เกิดข้อผิดพลาด: ${error.message}`);
            });
        }
    }).then((result) => {
        if (result.isConfirmed && result.value) {
            Swal.fire({
                icon: 'success',
                title: 'ลบผู้ใช้สำเร็จ!',
                text: result.value.message,
                confirmButtonColor: '#FF6B35'
            }).then(() => {
                location.reload();
            });
        }
    });
}

function initializeResponsive() {
    let resizeTimer;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(adjustLayout, 250);
    });
    adjustLayout();
}

function adjustLayout() {
    const width = window.innerWidth;
    if (width <= 768) {
        document.querySelectorAll('.action-btn span').forEach(el => {
            el.style.display = 'none';
        });
    } else {
        document.querySelectorAll('.action-btn span').forEach(el => {
            el.style.display = 'inline';
        });
    }
}

function initializeFilters() {
    document.querySelectorAll('select[name="role"], select[name="status"]').forEach(select => {
        select.addEventListener('change', function() {
            this.closest('form').submit();
        });
    });
}

function initializeTableInteractions() {
    const rows = document.querySelectorAll('.user-row');
    rows.forEach(row => {
        row.addEventListener('click', function(e) {
            if (e.target.closest('.action-btn, .table-checkbox')) {
                return;
            }
            const checkbox = this.querySelector('.user-checkbox');
            if (checkbox) {
                checkbox.checked = !checkbox.checked;
                updateBulkActions();
                updateSelectAllState();
            }
        });
    });
}

function setupEventListeners() {
    document.addEventListener('keydown', function(e) {
        if ((e.ctrlKey || e.metaKey) && e.key === 'a' && !e.target.matches('input, textarea, select')) {
            e.preventDefault();
            const selectAll = document.getElementById('selectAll');
            if (selectAll) {
                selectAll.checked = true;
                selectAll.dispatchEvent(new Event('change'));
            }
        }
        if (e.key === 'Escape') {
            clearSelection();
        }
    });
}

// Session messages
@if(session('success'))
    document.addEventListener('DOMContentLoaded', function() {
        showToast('success', '{{ session('success') }}');
    });
@endif

@if(session('error'))
    document.addEventListener('DOMContentLoaded', function() {
        Swal.fire({
            icon: 'error',
            title: 'เกิดข้อผิดพลาด!',
            text: '{{ session('error') }}',
            confirmButtonColor: '#FF6B35'
        });
    });
@endif
</script>
@endpush