@extends('layouts.app')

@section('title', 'เพิ่มคลังใหม่ - EazyCom Plus')
@section('page-title', 'เพิ่มคลังใหม่')
@section('mobile-title', 'เพิ่มคลัง')

@push('styles')
<style>
:root {
    --primary: #FF6B35;
    --primary-dark: #E85D2F;
    --primary-light: #FF8C61;
    --gradient: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
    --gradient-soft: linear-gradient(135deg, #FFE5DB 0%, #FFD4C4 100%);
    --success: #22C55E;
    --warning: #F59E0B;
    --danger: #EF4444;
    --info: #3B82F6;
    --purple: #8B5CF6;
    --white: #FFFFFF;
    --gray-50: #F8FAFC;
    --gray-100: #F1F5F9;
    --gray-200: #E2E8F0;
    --gray-400: #94A3B8;
    --gray-500: #64748B;
    --gray-600: #475569;
    --gray-700: #334155;
    --gray-900: #0F172A;
    --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
    --shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
    --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
    --radius-md: 12px;
    --radius-lg: 16px;
    --radius-xl: 20px;
    --transition: all 0.2s ease;
}

.create-page {
    padding: clamp(1rem, 4vw, 2rem);
    padding-bottom: clamp(2rem, 6vw, 4rem);
    background: var(--gray-50);
    min-height: 100vh;
}

.page-header {
    background: var(--white);
    border-radius: var(--radius-lg);
    padding: clamp(1.5rem, 4vw, 2rem);
    margin-bottom: clamp(1.5rem, 4vw, 2rem);
    box-shadow: var(--shadow);
    border: 1px solid var(--gray-200);
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 1.5rem;
    flex-wrap: wrap;
}

.header-info {
    flex: 1;
    min-width: 280px;
}

.header-title {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 0.75rem;
}

.title-icon {
    width: clamp(40px, 8vw, 48px);
    height: clamp(40px, 8vw, 48px);
    background: var(--gradient);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--white);
    font-size: clamp(1rem, 3vw, 1.25rem);
    box-shadow: var(--shadow-md);
}

.title-text h1 {
    font-size: clamp(1.5rem, 4vw, 1.875rem);
    font-weight: 700;
    color: var(--gray-900);
    margin: 0;
    line-height: 1.2;
}

.title-text p {
    font-size: clamp(0.875rem, 2.5vw, 1rem);
    color: var(--gray-600);
    margin: 0;
}

.header-actions {
    display: flex;
    gap: 0.75rem;
    flex-wrap: wrap;
}

.action-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: clamp(0.625rem, 2vw, 0.75rem) clamp(1rem, 3vw, 1.25rem);
    border-radius: var(--radius-md);
    font-weight: 600;
    font-size: clamp(0.8rem, 2vw, 0.875rem);
    text-decoration: none;
    border: none;
    cursor: pointer;
    transition: var(--transition);
    white-space: nowrap;
}

.back-btn {
    background: var(--white);
    color: var(--gray-600);
    border: 2px solid var(--gray-300);
}

.back-btn:hover {
    background: var(--gray-100);
    color: var(--gray-700);
    transform: translateY(-2px);
}

.form-container {
    background: var(--white);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow);
    border: 1px solid var(--gray-200);
    overflow: hidden;
}

.form-header {
    padding: clamp(1rem, 3vw, 1.25rem) clamp(1.25rem, 3vw, 1.5rem);
    background: var(--gradient-soft);
    border-bottom: 1px solid var(--gray-200);
}

.form-title {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 600;
    color: var(--gray-900);
    font-size: clamp(0.875rem, 2.5vw, 1rem);
}

.form-content {
    padding: clamp(1.5rem, 4vw, 2rem);
}

.form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: clamp(1.5rem, 4vw, 2rem);
    margin-bottom: clamp(2rem, 5vw, 3rem);
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.form-group.full-width {
    grid-column: 1 / -1;
}

.form-label {
    font-weight: 600;
    color: var(--gray-700);
    font-size: clamp(0.8rem, 2vw, 0.875rem);
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.form-label.required::after {
    content: '*';
    color: var(--danger);
    font-weight: bold;
}

.input-group {
    position: relative;
}

.input-icon {
    position: absolute;
    left: 0.75rem;
    top: 50%;
    transform: translateY(-50%);
    color: var(--gray-400);
    font-size: 0.875rem;
    z-index: 1;
}

.form-input {
    width: 100%;
    padding: clamp(0.75rem, 2vw, 1rem) clamp(2.5rem, 5vw, 3rem) clamp(0.75rem, 2vw, 1rem) clamp(0.75rem, 2vw, 1rem);
    border: 2px solid var(--gray-200);
    border-radius: var(--radius-md);
    font-size: clamp(0.875rem, 2vw, 1rem);
    transition: var(--transition);
    background: var(--white);
}

.form-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(255, 107, 53, 0.1);
}

.form-input.error {
    border-color: var(--danger);
}

.form-input.success {
    border-color: var(--success);
}

.form-textarea {
    resize: vertical;
    min-height: 100px;
    padding-top: clamp(0.75rem, 2vw, 1rem);
}

.radio-group {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-top: 0.5rem;
}

.radio-option {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: clamp(0.75rem, 2vw, 1rem);
    border: 2px solid var(--gray-200);
    border-radius: var(--radius-md);
    cursor: pointer;
    transition: var(--transition);
    background: var(--white);
}

.radio-option:hover {
    border-color: var(--primary);
    background: rgba(255, 107, 53, 0.02);
}

.radio-option.active {
    border-color: var(--primary);
    background: rgba(255, 107, 53, 0.05);
}

.radio-input {
    display: none;
}

.radio-custom {
    width: 20px;
    height: 20px;
    border: 2px solid var(--gray-300);
    border-radius: 50%;
    position: relative;
    transition: var(--transition);
    flex-shrink: 0;
}

.radio-option.active .radio-custom {
    border-color: var(--primary);
    background: var(--primary);
}

.radio-option.active .radio-custom::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 8px;
    height: 8px;
    background: var(--white);
    border-radius: 50%;
}

.radio-label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 500;
    color: var(--gray-700);
    font-size: clamp(0.875rem, 2vw, 1rem);
}

.info-card {
    background: linear-gradient(135deg, rgba(59, 130, 246, 0.05) 0%, rgba(147, 197, 253, 0.05) 100%);
    border: 1px solid rgba(59, 130, 246, 0.2);
    border-radius: var(--radius-lg);
    padding: clamp(1.25rem, 3vw, 1.5rem);
    margin-top: 1rem;
}

.info-header {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin-bottom: 1rem;
}

.info-icon {
    width: clamp(40px, 7vw, 48px);
    height: clamp(40px, 7vw, 48px);
    background: linear-gradient(135deg, var(--info) 0%, #1d4ed8 100%);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--white);
    font-size: clamp(1rem, 2.5vw, 1.25rem);
    flex-shrink: 0;
}

.info-title {
    font-size: clamp(1rem, 2.5vw, 1.125rem);
    font-weight: 600;
    color: var(--info);
}

.info-text {
    color: var(--info);
    margin-bottom: 0.75rem;
    font-size: clamp(0.875rem, 2vw, 1rem);
}

.info-list {
    list-style: none;
    padding: 0;
    margin: 0;
}

.info-list li {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: var(--info);
    font-size: clamp(0.8rem, 2vw, 0.875rem);
    margin-bottom: 0.5rem;
}

.info-list li::before {
    content: '✓';
    color: var(--success);
    font-weight: bold;
    width: 16px;
    text-align: center;
    flex-shrink: 0;
}

.error-message {
    color: var(--danger);
    font-size: clamp(0.75rem, 1.8vw, 0.875rem);
    margin-top: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.error-message::before {
    content: '⚠';
    color: var(--danger);
}

.help-text {
    color: var(--gray-500);
    font-size: clamp(0.75rem, 1.8vw, 0.875rem);
    margin-top: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.help-text::before {
    content: 'ℹ';
    color: var(--info);
    font-weight: bold;
}

.form-actions {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding-top: clamp(1.5rem, 4vw, 2rem);
    border-top: 2px solid var(--gray-200);
    margin-top: clamp(1.5rem, 4vw, 2rem);
    flex-wrap: wrap;
    gap: 1rem;
}

.progress-steps {
    display: flex;
    gap: clamp(1rem, 3vw, 2rem);
}

.progress-step {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.5rem;
    opacity: 0.4;
    transition: var(--transition);
}

.progress-step.active {
    opacity: 1;
}

.step-icon {
    width: clamp(36px, 6vw, 40px);
    height: clamp(36px, 6vw, 40px);
    background: var(--gray-200);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: clamp(0.875rem, 2vw, 1rem);
    color: var(--gray-500);
    transition: var(--transition);
}

.progress-step.active .step-icon {
    background: var(--gradient);
    color: var(--white);
}

.step-label {
    font-size: clamp(0.75rem, 1.8vw, 0.875rem);
    font-weight: 500;
    color: var(--gray-500);
    text-align: center;
}

.progress-step.active .step-label {
    color: var(--gray-700);
    font-weight: 600;
}

.action-buttons {
    display: flex;
    gap: 0.75rem;
    flex-wrap: wrap;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: clamp(0.75rem, 2vw, 1rem) clamp(1.25rem, 3vw, 1.5rem);
    border-radius: var(--radius-md);
    font-weight: 600;
    font-size: clamp(0.875rem, 2vw, 1rem);
    text-decoration: none;
    border: none;
    cursor: pointer;
    transition: var(--transition);
    position: relative;
    overflow: hidden;
}

.btn-primary {
    background: var(--gradient);
    color: var(--white);
    box-shadow: var(--shadow-md);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.btn-secondary {
    background: var(--white);
    color: var(--gray-600);
    border: 2px solid var(--gray-300);
}

.btn-secondary:hover {
    background: var(--gray-100);
    color: var(--gray-700);
}

.btn-loading {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    display: none;
}

.btn.loading .btn-loading {
    display: block;
}

.btn.loading > span:not(.btn-loading) {
    opacity: 0;
}

@media (max-width: 768px) {
    .header-content {
        flex-direction: column;
        align-items: stretch;
    }
    
    .header-actions {
        justify-content: stretch;
    }
    
    .action-btn {
        flex: 1;
        justify-content: center;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }
    
    .radio-group {
        grid-template-columns: 1fr;
    }
    
    .form-actions {
        flex-direction: column;
        align-items: stretch;
    }
    
    .progress-steps {
        display: none;
    }
    
    .action-buttons {
        justify-content: stretch;
    }
    
    .btn {
        flex: 1;
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .create-page {
        padding: 1rem;
        padding-bottom: 3rem;
    }
    
    .info-header {
        flex-direction: column;
        text-align: center;
    }
    
    .action-buttons {
        flex-direction: column;
    }
}

.loading {
    opacity: 0.6;
    pointer-events: none;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

.fade-in {
    animation: fadeIn 0.3s ease;
}
</style>
@endpush

@section('content')
<div class="create-page">
    <!-- Header Section -->
    <div class="page-header">
        <div class="header-content">
            <div class="header-info">
                <div class="header-title">
                    <div class="title-icon">
                        <i class="fas fa-plus-circle"></i>
                    </div>
                    <div class="title-text">
                        <h1>เพิ่มคลังใหม่</h1>
                        <p>เพิ่มข้อมูลคลังสินค้าใหม่เข้าสู่ระบบ</p>
                    </div>
                </div>
            </div>
            <div class="header-actions">
                <a href="{{ route('admin.warehouses.index') }}" class="action-btn back-btn">
                    <i class="fas fa-arrow-left"></i>
                    <span>กลับ</span>
                </a>
            </div>
        </div>
    </div>

    <!-- Form Container -->
    <div class="form-container">
        <div class="form-header">
            <div class="form-title">
                <i class="fas fa-warehouse"></i>
                <span>ข้อมูลคลังสินค้า</span>
            </div>
        </div>
        
        <div class="form-content">
            <form method="POST" action="{{ route('admin.warehouses.store') }}" id="warehouseForm">
                @csrf
                
                <div class="form-grid">
                    <!-- รหัสคลัง -->
                    <div class="form-group">
                        <label for="code" class="form-label required">รหัสคลัง</label>
                        <div class="input-group">
                            <div class="input-icon">
                                <i class="fas fa-barcode"></i>
                            </div>
                            <input type="text" id="code" name="code" value="{{ old('code') }}" 
                                   class="form-input @error('code') error @enderror" 
                                   placeholder="เช่น WH001, BKK01" required maxlength="10">
                        </div>
                        @error('code')
                            <div class="error-message">{{ $message }}</div>
                        @enderror
                        <div class="help-text">รหัสคลังต้องไม่ซ้ำกัน และมีได้สูงสุด 10 ตัวอักษร</div>
                    </div>

                    <!-- ชื่อคลัง -->
                    <div class="form-group">
                        <label for="name" class="form-label required">ชื่อคลัง</label>
                        <div class="input-group">
                            <div class="input-icon">
                                <i class="fas fa-warehouse"></i>
                            </div>
                            <input type="text" id="name" name="name" value="{{ old('name') }}" 
                                   class="form-input @error('name') error @enderror" 
                                   placeholder="เช่น คลังสินค้ากรุงเทพ" required>
                        </div>
                        @error('name')
                            <div class="error-message">{{ $message }}</div>
                        @enderror
                    </div>

                    <!-- ที่อยู่ -->
                    <div class="form-group full-width">
                        <label for="location" class="form-label">ที่อยู่</label>
                        <div class="input-group">
                            <div class="input-icon">
                                <i class="fas fa-map-marker-alt"></i>
                            </div>
                            <textarea id="location" name="location" rows="3" 
                                      class="form-input form-textarea @error('location') error @enderror" 
                                      placeholder="ระบุที่อยู่ของคลังสินค้า">{{ old('location') }}</textarea>
                        </div>
                        @error('location')
                            <div class="error-message">{{ $message }}</div>
                        @enderror
                    </div>

                    <!-- สถานะ -->
                    <div class="form-group full-width">
                        <label class="form-label">สถานะ</label>
                        <div class="radio-group">
                            <label class="radio-option active">
                                <input type="radio" name="is_active" value="1" class="radio-input"
                                       {{ old('is_active', '1') === '1' ? 'checked' : '' }}>
                                <div class="radio-custom"></div>
                                <div class="radio-label">
                                    <i class="fas fa-check-circle" style="color: var(--success);"></i>
                                    ใช้งานอยู่
                                </div>
                            </label>
                            <label class="radio-option">
                                <input type="radio" name="is_active" value="0" class="radio-input"
                                       {{ old('is_active') === '0' ? 'checked' : '' }}>
                                <div class="radio-custom"></div>
                                <div class="radio-label">
                                    <i class="fas fa-times-circle" style="color: var(--danger);"></i>
                                    ปิดใช้งาน
                                </div>
                            </label>
                        </div>
                    </div>

                    <!-- ข้อมูลเพิ่มเติม -->
                    <div class="form-group full-width">
                        <div class="info-card">
                            <div class="info-header">
                                <div class="info-icon">
                                    <i class="fas fa-info-circle"></i>
                                </div>
                                <div class="info-title">ข้อมูลเพิ่มเติม</div>
                            </div>
                            <div class="info-text">หลังจากสร้างคลังแล้ว คุณสามารถ:</div>
                            <ul class="info-list">
                                <li>กำหนดอัตราค่าคอมมิชชั่นสำหรับคลังนี้</li>
                                <li>กำหนดผู้ใช้งานให้กับคลัง</li>
                                <li>ติดตามสถิติการใช้งาน</li>
                            </ul>
                        </div>
                    </div>
                </div>

                <!-- Form Actions -->
                <div class="form-actions">
                    <div class="progress-steps">
                        <div class="progress-step active">
                            <div class="step-icon">
                                <i class="fas fa-warehouse"></i>
                            </div>
                            <div class="step-label">ข้อมูลคลัง</div>
                        </div>
                        <div class="progress-step">
                            <div class="step-icon">
                                <i class="fas fa-percentage"></i>
                            </div>
                            <div class="step-label">อัตราคอมมิชชั่น</div>
                        </div>
                        <div class="progress-step">
                            <div class="step-icon">
                                <i class="fas fa-users"></i>
                            </div>
                            <div class="step-label">ผู้ใช้งาน</div>
                        </div>
                    </div>
                    <div class="action-buttons">
                        <a href="{{ route('admin.warehouses.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times"></i>
                            <span>ยกเลิก</span>
                        </a>
                        <button type="submit" class="btn btn-primary" id="submitBtn">
                            <i class="fas fa-save"></i>
                            <span>บันทึก</span>
                            <div class="btn-loading">
                                <i class="fas fa-spinner fa-spin"></i>
                            </div>
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    initializeForm();
    setupValidation();
    setupInteractions();
});

function initializeForm() {
    const codeInput = document.getElementById('code');
    if (codeInput) {
        codeInput.addEventListener('input', function() {
            this.value = this.value.toUpperCase().replace(/[^A-Z0-9]/g, '');
            validateField(this);
        });
    }
    
    const radioOptions = document.querySelectorAll('.radio-option');
    radioOptions.forEach(option => {
        option.addEventListener('click', function() {
            const radio = this.querySelector('.radio-input');
            if (radio) {
                radio.checked = true;
                updateRadioStates();
            }
        });
    });
    
    updateRadioStates();
}

function setupValidation() {
    const form = document.getElementById('warehouseForm');
    
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        
        if (validateForm()) {
            submitForm();
        }
    });
    
    const inputs = form.querySelectorAll('input[required], textarea[required]');
    inputs.forEach(input => {
        input.addEventListener('blur', () => validateField(input));
        input.addEventListener('input', () => clearFieldError(input));
    });
}

function setupInteractions() {
    const formInputs = document.querySelectorAll('.form-input');
    formInputs.forEach(input => {
        input.addEventListener('focus', function() {
            this.parentElement.classList.add('focused');
        });
        
        input.addEventListener('blur', function() {
            this.parentElement.classList.remove('focused');
        });
    });
}

function validateField(field) {
    const value = field.value.trim();
    let isValid = true;
    let message = '';
    
    if (field.hasAttribute('required') && !value) {
        isValid = false;
        message = 'กรุณากรอกข้อมูลในช่องนี้';
    }
    
    if (field.id === 'code' && value) {
        if (value.length < 2) {
            isValid = false;
            message = 'รหัสคลังต้องมีอย่างน้อย 2 ตัวอักษร';
        } else if (value.length > 10) {
            isValid = false;
            message = 'รหัสคลังต้องมีไม่เกิน 10 ตัวอักษร';
        }
    }
    
    if (field.id === 'name' && value && value.length < 3) {
        isValid = false;
        message = 'ชื่อคลังต้องมีอย่างน้อย 3 ตัวอักษร';
    }
    
    if (isValid) {
        field.classList.remove('error');
        field.classList.add('success');
        removeFieldError(field);
    } else {
        field.classList.remove('success');
        field.classList.add('error');
        showFieldError(field, message);
    }
    
    return isValid;
}

function clearFieldError(field) {
    field.classList.remove('error', 'success');
    removeFieldError(field);
}

function showFieldError(field, message) {
    removeFieldError(field);
    
    const errorElement = document.createElement('div');
    errorElement.className = 'error-message';
    errorElement.textContent = message;
    
    field.parentElement.parentElement.appendChild(errorElement);
}

function removeFieldError(field) {
    const errorElement = field.parentElement.parentElement.querySelector('.error-message');
    if (errorElement) {
        errorElement.remove();
    }
}

function validateForm() {
    const code = document.getElementById('code').value.trim();
    const name = document.getElementById('name').value.trim();
    
    if (!code) {
        Swal.fire({
            icon: 'error',
            title: 'ข้อมูลไม่ครบถ้วน',
            text: 'กรุณาระบุรหัสคลัง',
            confirmButtonColor: '#FF6B35'
        });
        document.getElementById('code').focus();
        return false;
    }
    
    if (code.length < 2) {
        Swal.fire({
            icon: 'error',
            title: 'รหัสคลังไม่ถูกต้อง',
            text: 'รหัสคลังต้องมีอย่างน้อย 2 ตัวอักษร',
            confirmButtonColor: '#FF6B35'
        });
        document.getElementById('code').focus();
        return false;
    }
    
    if (!name) {
        Swal.fire({
            icon: 'error',
            title: 'ข้อมูลไม่ครบถ้วน',
            text: 'กรุณาระบุชื่อคลัง',
            confirmButtonColor: '#FF6B35'
        });
        document.getElementById('name').focus();
        return false;
    }
    
    if (name.length < 3) {
        Swal.fire({
            icon: 'error',
            title: 'ชื่อคลังไม่ถูกต้อง',
            text: 'ชื่อคลังต้องมีอย่างน้อย 3 ตัวอักษร',
            confirmButtonColor: '#FF6B35'
        });
        document.getElementById('name').focus();
        return false;
    }
    
    return true;
}

function submitForm() {
    const submitBtn = document.getElementById('submitBtn');
    const form = document.getElementById('warehouseForm');
    
    submitBtn.classList.add('loading');
    submitBtn.disabled = true;
    
    Swal.fire({
        title: 'ยืนยันการบันทึก',
        text: 'คุณต้องการบันทึกข้อมูลคลังใหม่นี้หรือไม่?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#FF6B35',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'บันทึก',
        cancelButtonText: 'ยกเลิก',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return new Promise((resolve) => {
                setTimeout(() => {
                    form.submit();
                    resolve();
                }, 1000);
            });
        }
    }).then((result) => {
        if (!result.isConfirmed) {
            submitBtn.classList.remove('loading');
            submitBtn.disabled = false;
        }
    });
}

function updateRadioStates() {
    const radioOptions = document.querySelectorAll('.radio-option');
    radioOptions.forEach(option => {
        const radio = option.querySelector('.radio-input');
        if (radio && radio.checked) {
            option.classList.add('active');
        } else {
            option.classList.remove('active');
        }
    });
}

@if(session('success'))
    document.addEventListener('DOMContentLoaded', function() {
        Swal.fire({
            icon: 'success',
            title: 'สำเร็จ!',
            text: '{{ session('success') }}',
            confirmButtonColor: '#FF6B35'
        });
    });
@endif

@if(session('error'))
    document.addEventListener('DOMContentLoaded', function() {
        Swal.fire({
            icon: 'error',
            title: 'เกิดข้อผิดพลาด!',
            text: '{{ session('error') }}',
            confirmButtonColor: '#FF6B35'
        });
    });
@endif

document.addEventListener('keydown', function(e) {
    if ((e.ctrlKey || e.metaKey) && e.key === 's') {
        e.preventDefault();
        if (validateForm()) {
            submitForm();
        }
    }
    
    if (e.key === 'Escape') {
        window.location.href = '{{ route('admin.warehouses.index') }}';
    }
});
</script>
@endpush