@extends('layouts.app')

@section('title', isset($record) ? 'แก้ไขรายการ' : 'เพิ่มรายการใหม่')
@section('page-title', isset($record) ? 'แก้ไขรายการ' : 'เพิ่มรายการใหม่')
@section('mobile-title', isset($record) ? 'แก้ไข' : 'เพิ่มใหม่')

@section('content')
<!-- Header Card -->
<div class="header-card">
    <div class="header-content">
        <div class="header-text">
            <div class="header-title">
                <i class="fas {{ isset($record) ? 'fa-edit' : 'fa-plus-circle' }}"></i>
                {{ isset($record) ? 'แก้ไขรายการ' : 'เพิ่มรายการใหม่' }}
            </div>
            <div class="header-subtitle">{{ isset($record) ? 'แก้ไขข้อมูลรายการการเงิน' : 'เพิ่มรายการรายรับหรือรายจ่ายใหม่' }}</div>
        </div>
        <div class="header-icon">
            <i class="fas fa-{{ isset($record) ? 'edit' : 'plus' }}"></i>
        </div>
    </div>
</div>

<form method="POST" 
      action="{{ isset($record) ? route('finance.records.update', $record) : route('finance.records.store') }}"
      id="financeForm">
    @csrf
    @if(isset($record))
        @method('PUT')
    @endif

    <!-- Type Selection Card -->
    @if(!isset($record))
    <div class="type-card">
        <div class="type-header">
            <div class="type-title">
                <i class="fas fa-exchange-alt"></i>
                เลือกประเภทรายการ
            </div>
        </div>
        <div class="type-body">
            <div class="type-options">
                <label class="type-option">
                    <input type="radio" name="type" value="income" {{ $type === 'income' ? 'checked' : '' }}>
                    <div class="type-selector income">
                        <div class="type-icon">
                            <i class="fas fa-arrow-up"></i>
                        </div>
                        <div class="type-info">
                            <div class="type-name">รายรับ</div>
                            <div class="type-desc">เงินเข้า</div>
                        </div>
                    </div>
                </label>
                <label class="type-option">
                    <input type="radio" name="type" value="expense" {{ $type === 'expense' ? 'checked' : '' }}>
                    <div class="type-selector expense">
                        <div class="type-icon">
                            <i class="fas fa-arrow-down"></i>
                        </div>
                        <div class="type-info">
                            <div class="type-name">รายจ่าย</div>
                            <div class="type-desc">เงินออก</div>
                        </div>
                    </div>
                </label>
            </div>
        </div>
    </div>
    @else
        <input type="hidden" name="type" value="{{ $record->type }}">
        <!-- Current Type Display -->
        <div class="current-type-card">
            <div class="current-type-content">
                <div class="current-type-icon {{ $record->type }}">
                    <i class="fas fa-{{ $record->type == 'income' ? 'arrow-up' : 'arrow-down' }}"></i>
                </div>
                <div class="current-type-info">
                    <div class="current-type-label">ประเภทรายการ</div>
                    <div class="current-type-name">{{ $record->type == 'income' ? 'รายรับ' : 'รายจ่าย' }}</div>
                </div>
            </div>
        </div>
    @endif

    <!-- Form Card -->
    <div class="form-card">
        <div class="form-header">
            <div class="form-title">
                <i class="fas fa-edit"></i>
                ข้อมูลรายการ
            </div>
        </div>
        <div class="form-body">
            <div class="form-grid">
                <!-- Date -->
                <div class="form-group">
                    <label class="form-label required">วันที่</label>
                    <div class="input-group">
                        <div class="input-icon">
                            <i class="fas fa-calendar-alt"></i>
                        </div>
                        <input type="date" 
                               name="record_date" 
                               class="form-input @error('record_date') error @enderror"
                               value="{{ old('record_date', isset($record) ? $record->record_date->format('Y-m-d') : date('Y-m-d')) }}"
                               required>
                    </div>
                    @error('record_date')
                        <div class="error-text">{{ $message }}</div>
                    @enderror
                </div>

                <!-- Category -->
                <div class="form-group">
                    <label class="form-label required">หมวดหมู่</label>
                    <div class="input-group">
                        <div class="input-icon">
                            <i class="fas fa-tag"></i>
                        </div>
                        <select name="category" 
                                class="form-input category-select @error('category') error @enderror"
                                required>
                            <option value="">-- เลือกหมวดหมู่ --</option>
                            @foreach($categories as $value => $name)
                                <option value="{{ $value }}" 
                                        {{ old('category', isset($record) ? $record->category : '') == $value ? 'selected' : '' }}>
                                    {{ $name }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    @error('category')
                        <div class="error-text">{{ $message }}</div>
                    @enderror
                </div>

                <!-- Amount -->
                <div class="form-group">
                    <label class="form-label required">จำนวนเงิน</label>
                    <div class="amount-input-group">
                        <div class="currency-symbol">฿</div>
                        <input type="number" 
                               name="amount" 
                               class="amount-input @error('amount') error @enderror"
                               value="{{ old('amount', isset($record) ? $record->amount : '') }}"
                               step="0.01"
                               min="0"
                               placeholder="0.00"
                               required>
                    </div>
                    @error('amount')
                        <div class="error-text">{{ $message }}</div>
                    @enderror
                    <div class="amount-display" id="amountDisplay"></div>
                </div>

                <!-- Note -->
                <div class="form-group full-width">
                    <label class="form-label">หมายเหตุ</label>
                    <div class="textarea-group">
                        <div class="textarea-icon">
                            <i class="fas fa-sticky-note"></i>
                        </div>
                        <textarea name="note" 
                                  class="form-textarea @error('note') error @enderror"
                                  rows="3"
                                  placeholder="รายละเอียดเพิ่มเติม (ถ้ามี)"
                                  maxlength="500">{{ old('note', isset($record) ? $record->note : '') }}</textarea>
                        <div class="char-counter">
                            <span id="charCount">{{ strlen(old('note', isset($record) ? $record->note : '')) }}</span>/500
                        </div>
                    </div>
                    @error('note')
                        <div class="error-text">{{ $message }}</div>
                    @enderror
                </div>

                <!-- Recurring (for create only) -->
                @if(!isset($record))
                <div class="form-group full-width">
                    <div class="toggle-group">
                        <label class="toggle-label">
                            <input type="checkbox" 
                                   name="is_recurring" 
                                   value="1"
                                   id="recurringCheck"
                                   class="toggle-input"
                                   {{ old('is_recurring') ? 'checked' : '' }}>
                            <div class="toggle-switch"></div>
                            <span class="toggle-text">รายการซ้ำ (สร้างรายการล่วงหน้า)</span>
                        </label>
                    </div>
                    
                    <div id="recurringOptions" class="recurring-options" style="display: {{ old('is_recurring') ? 'block' : 'none' }}">
                        <div class="recurring-select">
                            <select name="recurring_type" class="form-input">
                                <option value="daily" {{ old('recurring_type') == 'daily' ? 'selected' : '' }}>ทุกวัน</option>
                                <option value="weekly" {{ old('recurring_type') == 'weekly' ? 'selected' : '' }}>ทุกสัปดาห์</option>
                                <option value="monthly" {{ old('recurring_type') == 'monthly' ? 'selected' : '' }}>ทุกเดือน</option>
                                <option value="yearly" {{ old('recurring_type') == 'yearly' ? 'selected' : '' }}>ทุกปี</option>
                            </select>
                        </div>
                        <div class="recurring-hint">ระบบจะสร้างรายการล่วงหน้า 3 เดือน</div>
                    </div>
                </div>
                @endif
            </div>
        </div>
    </div>

</form>

<!-- Action Buttons -->
<div class="action-buttons">
    <button type="button" class="btn-secondary" onclick="window.history.back()">
        <i class="fas fa-arrow-left"></i>
        ยกเลิก
    </button>
    <button type="submit" class="btn-primary" form="financeForm">
        <i class="fas fa-save"></i>
        {{ isset($record) ? 'บันทึกการแก้ไข' : 'บันทึกรายการ' }}
    </button>
</div>

<!-- Category Preview -->
@if(!isset($record))
<div class="category-card">
    <div class="category-header">
        <div class="category-title">
            <i class="fas fa-tags"></i>
            หมวดหมู่ที่มี
        </div>
    </div>
    <div class="category-body">
        <div id="categoryList" class="category-list">
            <!-- Will be populated by JS -->
        </div>
    </div>
</div>
@endif

@push('styles')
<style>
/* Header Card */
.header-card {
    background: var(--gradient);
    border-radius: 20px;
    padding: 20px;
    margin-bottom: 20px;
    color: white;
}

.header-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-title {
    font-size: 20px;
    font-weight: 600;
    margin-bottom: 4px;
    display: flex;
    align-items: center;
    gap: 8px;
}

.header-subtitle {
    font-size: 14px;
    opacity: 0.9;
}

.header-icon {
    width: 60px;
    height: 60px;
    background: rgba(255,255,255,0.2);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
}

/* Type Card */
.type-card {
    background: white;
    border-radius: 16px;
    margin-bottom: 20px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
    overflow: hidden;
}

.type-header {
    background: var(--gradient-soft);
    padding: 16px 20px;
    border-bottom: 1px solid rgba(255,107,53,0.1);
}

.type-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary);
    display: flex;
    align-items: center;
    gap: 8px;
}

.type-body {
    padding: 20px;
}

.type-options {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 16px;
}

.type-option {
    cursor: pointer;
}

.type-option input[type="radio"] {
    display: none;
}

.type-selector {
    display: flex;
    align-items: center;
    padding: 16px;
    border: 2px solid var(--gray-200);
    border-radius: 12px;
    transition: all 0.3s;
    background: white;
}

.type-selector:hover {
    border-color: var(--gray-300);
    transform: translateY(-1px);
}

.type-option input[type="radio"]:checked + .type-selector {
    border-color: var(--primary);
    background: rgba(255,107,53,0.05);
}

.type-icon {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
    margin-right: 12px;
}

.type-selector.income .type-icon {
    background: rgba(34,197,94,0.1);
    color: #22C55E;
}

.type-selector.expense .type-icon {
    background: rgba(239,68,68,0.1);
    color: #EF4444;
}

.type-info {
    flex: 1;
}

.type-name {
    font-size: 16px;
    font-weight: 600;
    color: var(--gray-900);
    margin-bottom: 2px;
}

.type-desc {
    font-size: 12px;
    color: var(--gray-500);
}

/* Current Type Display */
.current-type-card {
    background: white;
    border-radius: 16px;
    padding: 20px;
    margin-bottom: 20px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
}

.current-type-content {
    display: flex;
    align-items: center;
    gap: 16px;
}

.current-type-icon {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
}

.current-type-icon.income {
    background: rgba(34,197,94,0.1);
    color: #22C55E;
}

.current-type-icon.expense {
    background: rgba(239,68,68,0.1);
    color: #EF4444;
}

.current-type-label {
    font-size: 12px;
    color: var(--gray-500);
    margin-bottom: 2px;
}

.current-type-name {
    font-size: 16px;
    font-weight: 600;
    color: var(--gray-900);
}

/* Form Card */
.form-card {
    background: white;
    border-radius: 16px;
    margin-bottom: 20px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
    overflow: hidden;
}

.form-header {
    background: var(--gradient-soft);
    padding: 16px 20px;
    border-bottom: 1px solid rgba(255,107,53,0.1);
}

.form-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary);
    display: flex;
    align-items: center;
    gap: 8px;
}

.form-body {
    padding: 20px;
}

.form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group.full-width {
    grid-column: 1 / -1;
}

.form-label {
    font-size: 14px;
    font-weight: 500;
    color: var(--gray-900);
    margin-bottom: 8px;
}

.form-label.required::after {
    content: ' *';
    color: #EF4444;
}

.input-group {
    position: relative;
}

.input-icon {
    position: absolute;
    left: 16px;
    top: 50%;
    transform: translateY(-50%);
    color: var(--gray-400);
    z-index: 2;
}

.form-input {
    width: 100%;
    padding: 16px 16px 16px 48px;
    border: 2px solid var(--gray-200);
    border-radius: 12px;
    font-size: 16px;
    font-family: inherit;
    transition: all 0.3s;
    background: white;
}

.form-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(255,107,53,0.1);
}

.form-input.error {
    border-color: #EF4444;
}

.error-text {
    color: #EF4444;
    font-size: 12px;
    margin-top: 4px;
}

/* Amount Input */
.amount-input-group {
    position: relative;
    display: flex;
    align-items: center;
}

.currency-symbol {
    position: absolute;
    left: 16px;
    top: 50%;
    transform: translateY(-50%);
    font-size: 18px;
    font-weight: 600;
    color: var(--primary);
    z-index: 2;
}

.amount-input {
    width: 100%;
    padding: 16px 16px 16px 40px;
    border: 2px solid var(--gray-200);
    border-radius: 12px;
    font-size: 18px;
    font-weight: 600;
    text-align: right;
    font-family: inherit;
    transition: all 0.3s;
    background: white;
}

.amount-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(255,107,53,0.1);
}

.amount-input.error {
    border-color: #EF4444;
}

.amount-display {
    margin-top: 8px;
    font-size: 14px;
    color: var(--gray-600);
    text-align: right;
}

/* Textarea */
.textarea-group {
    position: relative;
}

.textarea-icon {
    position: absolute;
    left: 16px;
    top: 16px;
    color: var(--gray-400);
    z-index: 2;
}

.form-textarea {
    width: 100%;
    padding: 16px 16px 16px 48px;
    border: 2px solid var(--gray-200);
    border-radius: 12px;
    font-size: 16px;
    font-family: inherit;
    transition: all 0.3s;
    background: white;
    resize: vertical;
    min-height: 80px;
}

.form-textarea:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(255,107,53,0.1);
}

.form-textarea.error {
    border-color: #EF4444;
}

.char-counter {
    position: absolute;
    bottom: 8px;
    right: 12px;
    font-size: 12px;
    color: var(--gray-400);
    background: white;
    padding: 2px 4px;
    border-radius: 4px;
}

/* Toggle Switch */
.toggle-group {
    margin-bottom: 16px;
}

.toggle-label {
    display: flex;
    align-items: center;
    gap: 12px;
    cursor: pointer;
    font-weight: 500;
    color: var(--gray-900);
}

.toggle-input {
    display: none;
}

.toggle-switch {
    width: 48px;
    height: 24px;
    background: var(--gray-300);
    border-radius: 12px;
    position: relative;
    transition: all 0.3s;
}

.toggle-switch::after {
    content: '';
    width: 20px;
    height: 20px;
    background: white;
    border-radius: 50%;
    position: absolute;
    top: 2px;
    left: 2px;
    transition: all 0.3s;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.toggle-input:checked + .toggle-switch {
    background: var(--primary);
}

.toggle-input:checked + .toggle-switch::after {
    transform: translateX(24px);
}

.recurring-options {
    margin-top: 16px;
    padding: 16px;
    background: rgba(255,107,53,0.05);
    border-radius: 12px;
    border: 1px solid rgba(255,107,53,0.1);
}

.recurring-select {
    margin-bottom: 8px;
}

.recurring-hint {
    font-size: 12px;
    color: var(--gray-600);
}

/* Category Card */
.category-card {
    background: white;
    border-radius: 16px;
    margin-bottom: 20px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
    overflow: hidden;
}

.category-header {
    background: var(--gradient-soft);
    padding: 16px 20px;
    border-bottom: 1px solid rgba(255,107,53,0.1);
}

.category-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary);
    display: flex;
    align-items: center;
    gap: 8px;
}

.category-body {
    padding: 20px;
}

.category-list {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
}

.category-item {
    display: flex;
    align-items: center;
    gap: 6px;
    padding: 8px 12px;
    background: var(--gray-50);
    border-radius: 20px;
    font-size: 12px;
    font-weight: 500;
    color: var(--gray-700);
    border: 1px solid var(--gray-200);
}

/* Action Buttons */
.action-buttons {
    display: flex;
    gap: 12px;
    margin-bottom: 100px;
}

.btn-primary,
.btn-secondary {
    flex: 1;
    padding: 16px 20px;
    border-radius: 12px;
    font-size: 16px;
    font-weight: 600;
    border: none;
    cursor: pointer;
    transition: all 0.3s;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
}

.btn-primary {
    background: var(--gradient);
    color: white;
    box-shadow: 0 4px 12px rgba(255,107,53,0.3);
}

.btn-primary:active {
    transform: scale(0.98);
}

.btn-secondary {
    background: var(--gray-100);
    color: var(--gray-700);
    border: 2px solid var(--gray-200);
}

.btn-secondary:active {
    transform: scale(0.98);
    background: var(--gray-200);
}

/* Desktop Styles */
@media (min-width: 769px) {
    .header-card {
        display: none;
    }
    
    .action-buttons {
        justify-content: flex-end;
    }
    
    .btn-primary,
    .btn-secondary {
        flex: none;
        min-width: 140px;
    }
}

@media (max-width: 768px) {
    .type-options {
        grid-template-columns: 1fr;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
    }
    
    .action-buttons {
        flex-direction: column;
    }
}
</style>
@endpush

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Type selector
    const typeInputs = document.querySelectorAll('input[name="type"]');
    const categorySelect = document.querySelector('.category-select');
    
    typeInputs.forEach(input => {
        input.addEventListener('change', function() {
            loadCategories(this.value);
        });
    });

    // Recurring toggle
    const recurringCheck = document.getElementById('recurringCheck');
    const recurringOptions = document.getElementById('recurringOptions');
    
    if (recurringCheck) {
        recurringCheck.addEventListener('change', function() {
            recurringOptions.style.display = this.checked ? 'block' : 'none';
        });
    }

    // Character counter
    const textarea = document.querySelector('.form-textarea');
    const charCount = document.getElementById('charCount');
    
    if (textarea && charCount) {
        textarea.addEventListener('input', function() {
            charCount.textContent = this.value.length;
        });
    }

    // Amount formatting and display
    const amountInput = document.querySelector('.amount-input');
    const amountDisplay = document.getElementById('amountDisplay');
    
    if (amountInput) {
        amountInput.addEventListener('input', function() {
            // Format input
            this.value = this.value.replace(/[^0-9.]/g, '');
            
            const parts = this.value.split('.');
            if (parts.length > 2) {
                this.value = parts[0] + '.' + parts.slice(1).join('');
            }
            
            if (parts[1] && parts[1].length > 2) {
                this.value = parts[0] + '.' + parts[1].substring(0, 2);
            }
            
            // Update display
            if (amountDisplay && this.value) {
                const amount = parseFloat(this.value);
                if (!isNaN(amount)) {
                    amountDisplay.textContent = new Intl.NumberFormat('th-TH', {
                        style: 'currency',
                        currency: 'THB'
                    }).format(amount);
                } else {
                    amountDisplay.textContent = '';
                }
            }
        });
        
        // Initial display
        if (amountInput.value && amountDisplay) {
            const amount = parseFloat(amountInput.value);
            if (!isNaN(amount)) {
                amountDisplay.textContent = new Intl.NumberFormat('th-TH', {
                    style: 'currency',
                    currency: 'THB'
                }).format(amount);
            }
        }
    }

    // Load categories based on type
    function loadCategories(type) {
        if (!categorySelect) return;
        
        fetch(`{{ route('finance.records.categories') }}?type=${type}`)
            .then(response => response.json())
            .then(categories => {
                categorySelect.innerHTML = '<option value="">-- เลือกหมวดหมู่ --</option>';
                categories.forEach(cat => {
                    const option = document.createElement('option');
                    option.value = cat.slug;
                    option.textContent = cat.name;
                    categorySelect.appendChild(option);
                });
                
                updateCategoryPreview(categories);
            })
            .catch(error => {
                console.error('Error loading categories:', error);
            });
    }

    // Update category preview
    function updateCategoryPreview(categories) {
        const categoryList = document.getElementById('categoryList');
        if (!categoryList) return;
        
        categoryList.innerHTML = '';
        
        categories.forEach(cat => {
            const item = document.createElement('div');
            item.className = 'category-item';
            item.innerHTML = `<i class="fas ${cat.icon || 'fa-tag'}"></i> ${cat.name}`;
            categoryList.appendChild(item);
        });
    }

    // Initial load
    const currentType = document.querySelector('input[name="type"]:checked');
    if (currentType) {
        loadCategories(currentType.value);
    }

    // Form validation
    const form = document.getElementById('financeForm');
    if (form) {
        form.addEventListener('submit', function(e) {
            const amount = amountInput?.value;
            if (amount && parseFloat(amount) <= 0) {
                e.preventDefault();
                alert('กรุณากรอกจำนวนเงินที่ถูกต้อง');
                amountInput.focus();
            }
        });
    }
});
</script>
@endpush
@endsection