<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Session;
use Tests\TestCase;
use App\Models\User;

class I18nTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create a test user
        $this->user = User::factory()->create([
            'role' => 'vehicle_user'
        ]);
    }

    /** @test */
    public function it_can_switch_language_via_ajax()
    {
        $this->actingAs($this->user);
        
        $response = $this->postJson('/language/switch', [
            'locale' => 'en'
        ]);
        
        $response->assertStatus(200)
                ->assertJson([
                    'success' => true,
                    'locale' => 'en'
                ]);
        
        $this->assertEquals('en', Session::get('locale'));
    }

    /** @test */
    public function it_can_switch_language_via_redirect()
    {
        $this->actingAs($this->user);
        
        $response = $this->post('/language/switch-redirect', [
            'locale' => 'th'
        ]);
        
        $response->assertRedirect();
        $this->assertEquals('th', Session::get('locale'));
    }

    /** @test */
    public function it_rejects_invalid_locale()
    {
        $this->actingAs($this->user);
        
        $response = $this->postJson('/language/switch', [
            'locale' => 'invalid'
        ]);
        
        $response->assertStatus(400)
                ->assertJson([
                    'success' => false
                ]);
    }

    /** @test */
    public function it_sets_locale_from_session()
    {
        Session::put('locale', 'en');
        
        $this->get('/dashboard');
        
        $this->assertEquals('en', App::getLocale());
    }

    /** @test */
    public function it_falls_back_to_default_locale()
    {
        Session::forget('locale');
        
        $this->get('/dashboard');
        
        $this->assertEquals(config('app.locale'), App::getLocale());
    }

    /** @test */
    public function it_can_get_translations_via_api()
    {
        $this->actingAs($this->user);
        
        $response = $this->get('/api/translations/th');
        
        $response->assertStatus(200)
                ->assertJsonStructure([
                    'common',
                    'delivery',
                    'finance'
                ]);
    }

    /** @test */
    public function it_can_clear_translation_cache()
    {
        $this->actingAs($this->user);
        
        $response = $this->post('/api/translations/clear-cache');
        
        $response->assertStatus(200)
                ->assertJson([
                    'success' => true
                ]);
    }

    /** @test */
    public function it_formats_currency_correctly()
    {
        App::setLocale('th');
        $formatted = \App\Helpers\LocaleHelper::formatCurrency(1234.56);
        $this->assertStringContains('฿', $formatted);
        
        App::setLocale('en');
        $formatted = \App\Helpers\LocaleHelper::formatCurrency(1234.56);
        $this->assertStringContains('$', $formatted);
    }

    /** @test */
    public function it_formats_numbers_correctly()
    {
        $formatted = \App\Helpers\LocaleHelper::formatNumber(1234567.89, 2);
        $this->assertEquals('1,234,567.89', $formatted);
    }

    /** @test */
    public function it_formats_dates_correctly()
    {
        $date = '2024-01-15';
        
        App::setLocale('th');
        $formatted = \App\Helpers\LocaleHelper::formatDate($date);
        $this->assertNotEmpty($formatted);
        
        App::setLocale('en');
        $formatted = \App\Helpers\LocaleHelper::formatDate($date);
        $this->assertNotEmpty($formatted);
    }

    /** @test */
    public function it_provides_correct_month_names()
    {
        App::setLocale('th');
        $months = \App\Helpers\LocaleHelper::getMonthNames();
        $this->assertEquals('มกราคม', $months[1]);
        
        App::setLocale('en');
        $months = \App\Helpers\LocaleHelper::getMonthNames();
        $this->assertEquals('January', $months[1]);
    }

    /** @test */
    public function it_provides_correct_day_names()
    {
        App::setLocale('th');
        $days = \App\Helpers\LocaleHelper::getDayNames();
        $this->assertEquals('อาทิตย์', $days[0]);
        
        App::setLocale('en');
        $days = \App\Helpers\LocaleHelper::getDayNames();
        $this->assertEquals('Sunday', $days[0]);
    }

    /** @test */
    public function it_handles_missing_translations_gracefully()
    {
        App::setLocale('th');
        $translation = __('non.existent.key');
        $this->assertEquals('non.existent.key', $translation);
    }

    /** @test */
    public function it_supports_translation_parameters()
    {
        App::setLocale('en');
        // Assuming we have a translation like "Hello :name"
        $translation = __('common.min_length', ['min' => 5]);
        $this->assertStringContains('5', $translation);
    }

    /** @test */
    public function blade_directives_work_correctly()
    {
        $this->actingAs($this->user);
        
        // Test currency directive
        $view = view('test-i18n', ['amount' => 1234.56])->render();
        // This would need a test view file to properly test
        
        $this->assertTrue(true); // Placeholder assertion
    }

    /** @test */
    public function locale_middleware_sets_correct_locale()
    {
        // Test query parameter
        $response = $this->get('/?lang=en');
        $this->assertEquals('en', App::getLocale());
        
        // Test cookie
        $response = $this->withCookie('locale', 'th')->get('/');
        $this->assertEquals('th', App::getLocale());
    }

    /** @test */
    public function pwa_manifest_serves_correct_language()
    {
        $response = $this->get('/manifest-th.json');
        $response->assertStatus(200);
        
        $manifest = json_decode($response->getContent(), true);
        $this->assertEquals('th', $manifest['lang']);
        $this->assertStringContains('ระบบจัดการ', $manifest['name']);
        
        $response = $this->get('/manifest-en.json');
        $response->assertStatus(200);
        
        $manifest = json_decode($response->getContent(), true);
        $this->assertEquals('en', $manifest['lang']);
        $this->assertStringContains('Commission Management', $manifest['name']);
    }
}